<?php

namespace Drupal\registered_organisations;

use Drupal\Component\Plugin\Factory\DefaultFactory;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Plugin\DefaultPluginManager;

/**
 * Service class for GovUK Notify.
 */
class OrganisationManager extends DefaultPluginManager implements OrganisationManagerInterface {

  /**
   * Constructs an Organisation object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache_backend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler to invoke the alter hook with.
   */
  public function __construct(\Traversable $namespaces, CacheBackendInterface $cache_backend, ModuleHandlerInterface $module_handler) {
    parent::__construct(
          'Plugin/OrganisationRegister',
          $namespaces,
          $module_handler,
          'Drupal\registered_organisations\OrganisationRegisterInterface',
          'Drupal\registered_organisations\Annotation\OrganisationRegister'
      );

    $this->alterInfo('registered_organisations_info');
    $this->setCacheBackend($cache_backend, 'registered_organisations_info_plugins');
    $this->factory = new DefaultFactory($this->getDiscovery());
  }

  /**
   * Get the register.
   *
   * @param string $register
   *   The plugin id of the register to use.
   *
   * @return OrganisationRegisterInterface
   *   The register plugin.
   */
  protected function getRegistry(string $register): OrganisationRegisterInterface {
    $definition = $this->getDefinition($register);
    return $this->createInstance($definition['id'], $definition);
  }

  /**
   * {@inheritDoc}
   */
  public function lookupOrganisation(string $register, string $id, bool $exception_on_failure = TRUE) {
    $profile = &drupal_static(__FUNCTION__ . ':' . $register . ':' . $id);
    if (isset($profile)) {
      return $profile;
    }

    return $this->getRegistry($register)?->getOrganisation($id);
  }

  /**
   * {@inheritDoc}
   */
  public function searchOrganisation(string $register, string $term, bool $exception_on_failure = TRUE): array {
    $search = &drupal_static(__FUNCTION__ . ':' . $register . ':' . $term);
    if (isset($search)) {
      return $search;
    }

    return $this->getRegistry($register)?->findOrganisation($term);
  }

}
