<?php

namespace Drupal\registered_organisations\Plugin\OrganisationRegister;

use Drupal\Core\Http\ClientFactory;
use Drupal\registered_organisations\DataException;
use Drupal\registered_organisations\RegisterException;
use Drupal\registered_organisations\Organisation;
use Drupal\registered_organisations\OrganisationInterface;
use Drupal\registered_organisations\OrganisationRegisterApi;
use Drupal\registered_organisations\OrganisationRegisterInterface;
use GuzzleHttp\ClientInterface;
use GuzzleHttp\Exception\ClientException;

/**
 * Charity Commission organisation register.
 *
 * @OrganisationRegister(
 *   id = "charity_commission",
 *   label = @Translation("Charity Commission"),
 *   description = @Translation("UK charities registered with the Charity Commission."),
 *   url = "https://www.gov.uk/government/organisations/charity-commission",
 *   search_url = "https://register-of-charities.charitycommission.gov.uk/charity-search"
 * )
 */
class CharityCommissionRegister extends OrganisationRegisterApi {

  /**
   * The organisation type enum.
   */
  const ORGANISATION_TYPE = [
    'Charitable company' => 'Charitable company',
    'CIO' => 'CIO',
    'Other' => 'Other',
    'Trust' => 'Trust',
  ];

  /**
   * The organisation status enum.
   */
  const ORGANISATION_STATUS = [
    'R' => "Registered",
    'RM' => "Removed",
  ];

  /**
   * The client for handling api requests.
   *
   * @var ClientInterface
   */
  protected mixed $client = NULL;

  /**
   * {@inheritDoc}
   */
  public function getClient() {
    $config = \Drupal::config('registered_organisations.settings');

    if (!$api_key = $config->get('charity_commission_api_key')) {
      throw new RegisterException("The Charity Commission API key is not configured.");
    }

    // If rate limiting has been hit wait for 5 minutes
    // before allowing requests to be retried.
    $cid = implode(':', [$this->getCachePrefix(), 'rate_limit']);
    if ($this->getCache()->get($cid)) {
      throw new RegisterException("Rate limit has been hit, please wait before retrying this operation.");
    }

    try {
      /* @var ClientFactory $factory */
      $factory = \Drupal::service('http_client_factory');
      return $factory->fromOptions([
        'base_uri' => 'https://api.charitycommission.gov.uk/register/api/',
        'headers' => [
          'Ocp-Apim-Subscription-Key' => $api_key,
        ],
      ]);
    }
    catch (\Exception $e) {
      throw new RegisterException("Failed to create Charity Commission API client");
    }
  }

  /**
   * {@inheritdoc}
   */
  public function process(string $key, mixed $value): mixed {
    // Convert the type and status to a readable format.
    switch ($key) {
      case 'type':
        return self::ORGANISATION_TYPE[$value] ?? $value;

        break;

      case 'status':
        return self::ORGANISATION_STATUS[$value] ?? $value;

        break;
    }

    return parent::process($key, $value);
  }

  /**
   * {@inheritDoc}
   */
  public function getOrganisation(string $id): ?OrganisationInterface {
    try {
      $response = $this->client->get('charityRegNumber/' . $id . '/0');
      $result = json_decode($response->getBody());
    }
    catch (\Throwable $e) {
      $this->handleException($e);
      return NULL;
    }

    // Construct the organisation profile object.
    $charity_information = [
      'type' => 'charity',
      'id' => $result->reg_charity_number,
      'name' => $result->charity_name,
      'status' => $result->reg_status,
    ];
    return new Organisation($this, $charity_information);
  }

  /**
   * {@inheritDoc}
   */
  public function findOrganisation(string $name): array {
    $organisations = [];

    try {
      $response = $this->client->get('searchCharityName' . '/' . $name);
      $results = json_decode($response->getBody());
    }
    catch (\Throwable $e) {
      $this->handleException($e);
      return $organisations;
    }

    // Create an organisation profile for each result.
    $organisations = [];
    foreach ($results as $result) {
      // If requested filter out in active charities.
      if ($result->reg_status != 'R') {
        continue;
      }

       $charity_information = [
        'type' => 'charity',
        'id' => $result->reg_charity_number,
        'name' => $result->charity_name,
        'status' => $result->reg_status,
      ];
      $organisations[$result->reg_charity_number] = new Organisation($this, $charity_information);

      // Limit results to 20 items.
      if (count($organisations) >= 20) {
        break;
      }
    }

    return $organisations;
  }

  /**
   * {@inheritdoc}
   */
  private function handleException(\Throwable $e) {
    switch (TRUE) {
      case $e instanceof ClientException:

        // Nothing found.
        if ($e->getCode() == 404) {
          throw new DataException("Charity information could not be found.");
        }

        // Rate limit exceeded.
        if ($e->getCode() == 429) {

          // Create lock record in cache.
          $cid = implode(':', [$this->getCachePrefix(), 'rate_limit']);
          $expiry = $this->getRequestTime() + OrganisationRegisterInterface::RATE_LIMIT_BUFFER;
          $this->getCache()->set($cid, $this->getRequestTime(), $expiry);

          throw new RegisterException("Too many requests, rate limit reached.");
        }

        throw new RegisterException("Charity Commission API Client exception encountered: @message");

      default:
        throw new RegisterException("Charity Commission API exception encountered: @message");
    }
  }

}
