<?php

namespace Drupal\related_content_block\Form;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;

/**
 * Related Content Settings form.
 */
class RelatedContentSettingsForm extends ConfigFormBase {

  /**
   * The field manager service.
   *
   * @var \Drupal\Core\Entity\EntityFieldManagerInterface
   */
  protected $fieldManager;

  /**
   * The entity display repository service.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface
   */
  protected $entityDisplayRepository;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Constructs a RelatedContentSettingsForm object.
   */
  public function __construct(EntityFieldManagerInterface $field_manager, EntityDisplayRepositoryInterface $entity_display_repository, EntityTypeManagerInterface $entity_type_manager) {
    $this->fieldManager = $field_manager;
    $this->entityDisplayRepository = $entity_display_repository;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_field.manager'),
      $container->get('entity_display.repository'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   *
   * Returns the list of configuration names that can be edited.
   */
  protected function getEditableConfigNames() {
    return ['related_content_block.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * Returns the unique form ID for this configuration form.
   */
  public function getFormId() {
    return 'related_content_block_settings_form';
  }

  /**
   * {@inheritdoc}
   *
   * @SuppressWarnings(camelCase)
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('related_content_block.settings');

    // Get all available content types (node bundles)
    $node_types = $this->entityTypeManager
      ->getStorage('node_type')
      ->loadMultiple();

    $bundles = [];
    foreach ($node_types as $machine_name => $type) {
      $bundles[$machine_name] = $type->label();
    }

    // Get view modes for selected node and bundle.
    $view_modes = $this->entityDisplayRepository->getViewModes('node');
    $view_modes_list = [];

    foreach ($view_modes as $machine_name => $info) {
      $view_modes_list[$machine_name] = $info['label'];
    }

    // Default values.
    $limited_pages = $config->get('limited_pages') ?? 3;
    $display = $config->get('display') ?? 'one_column';
    $selected_bundle = $form_state->getValue('selected_bundle') ?? $config->get('selected_bundle') ?? 'articles';
    $items_page = $config->get('items_per_page') ?? 6;
    $enable_owner_filter = $config->get('enable_owner_filter') ?? FALSE;

    // Default view mode value.
    $selected_view_mode = $form_state->getValue('selected_view_mode') ?? $config->get('selected_view_mode') ?? 'teaser';

    $form['selected_bundle'] = [
      '#type' => 'select',
      '#title' => $this->t('Select content type'),
      '#options' => $bundles,
      '#default_value' => $selected_bundle,
      '#ajax' => [
        'callback' => '::updateFieldOptions',
        'wrapper' => 'field-options-wrapper',
        'event' => 'change',
      ],
    ];

    $form['selected_view_mode'] = [
      '#type' => 'select',
      '#title' => $this->t('Select view mode'),
      '#options' => $view_modes_list,
      '#default_value' => $selected_view_mode,
      '#description' => $this->t('Choose the view mode to render related content.'),
    ];

    $form['field_options_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'field-options-wrapper'],
    ];

    // Get the fields of the selected content type.
    $fields = $this->fieldManager->getFieldDefinitions('node', $selected_bundle);
    $taxonomy_options = [];
    $profile_options = [];

    foreach ($fields as $field_name => $field_definition) {
      if ($field_definition->getType() === 'entity_reference') {
        $target = $field_definition->getSetting('target_type');
        if ($target === 'taxonomy_term') {
          $taxonomy_options[$field_name] = $field_definition->getLabel();
        }
        if ($target === 'profile') {
          $profile_options[$field_name] = $field_definition->getLabel();
        }
      }
    }

    $form['field_options_wrapper']['related_taxonomy_fields'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Related taxonomy fields'),
      '#options' => $taxonomy_options,
      '#default_value' => $config->get('related_taxonomy_fields') ?? [],
      '#description' => $this->t('Select taxonomy reference fields.'),
    ];

    $form['field_options_wrapper']['related_author_fields'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Related profile fields'),
      '#options' => $profile_options,
      '#default_value' => $config->get('related_author_fields') ?? [],
      '#description' => $this->t('Select profile reference fields.'),
    ];

    $form['enable_owner_filter'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Filter by node owner'),
      '#description' => $this->t('If enabled, only content from the same owner will be shown.'),
      '#default_value' => $enable_owner_filter,
    ];

    $form['display'] = [
      '#type' => 'radios',
      '#title' => $this->t('Display options'),
      '#options' => [
        'one_column' => $this->t('One column'),
        'two_columns' => $this->t('Two columns'),
      ],
      '#default_value' => $display,
      '#description' => $this->t('Select display mode for rendering items.'),
    ];

    $form['items_per_page'] = [
      '#type' => 'number',
      '#title' => $this->t('Items per page'),
      '#description' => $this->t('Maximum number of related nodes loaded per page.'),
      '#min' => 1,
      '#default_value' => $items_page,
      '#required' => TRUE,
    ];

    $form['limited_pages'] = [
      '#type' => 'number',
      '#title' => $this->t('Maximum pages to load'),
      '#description' => $this->t('Limit the number of load more pages available.'),
      '#min' => 1,
      '#default_value' => $limited_pages,
    ];

    return parent::buildForm($form, $form_state);

  }

  /**
   * AJAX callback to rebuild the field options container.
   */
  public function updateFieldOptions(array &$form, FormStateInterface $form_state) {
    $form_state->setRebuild(TRUE);
    return $form['field_options_wrapper'];
  }

  /**
   * {@inheritdoc}
   *
   * Form submission handler for the configuration form.
   *
   * Saves the configuration values and clears related cache tags.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('related_content_block.settings');

    $config->set('selected_bundle', $form_state->getValue('selected_bundle'))
      ->set('selected_view_mode', $form_state->getValue('selected_view_mode'))
      ->set('related_taxonomy_fields', array_filter($form_state->getValue('related_taxonomy_fields')))
      ->set('related_author_fields', array_filter($form_state->getValue('related_author_fields')))
      ->set('enable_owner_filter', (bool) $form_state->getValue('enable_owner_filter'))
      ->set('display', $form_state->getValue('display'))
      ->set('items_per_page', (int) $form_state->getValue('items_per_page'))
      ->set('limited_pages', (int) $form_state->getValue('limited_pages'))
      ->save();

    Cache::invalidateTags(['related_content_block_config', 'node_list']);

    parent::submitForm($form, $form_state);

  }

}
