<?php

namespace Drupal\related_content_block\Plugin\Block;

use Drupal\Core\Block\BlockBase;
use Drupal\node\NodeInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\related_content_block\RelatedContentService;

/**
 * Provides a 'Related Content' block.
 *
 * @Block(
 *   id = "related_content_block",
 *   admin_label = @Translation("Related Content"),
 *   category = @Translation("Custom")
 * )
 */
class RelatedContentBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The related content service.
   *
   * @var \Drupal\related_content_block\RelatedContentService
   */
  protected $relatedContentService;

  /**
   * Constructs a RelatedContentBlock.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entityTypeManager, ConfigFactoryInterface $configFactory, RouteMatchInterface $route_match, RelatedContentService $relatedContentService) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entityTypeManager;
    $this->configFactory = $configFactory;
    $this->routeMatch = $route_match;
    $this->relatedContentService = $relatedContentService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('config.factory'),
      $container->get('current_route_match'),
      $container->get('related_content_block.service')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function build(): array {
    $node = $this->routeMatch->getParameter('node');

    if (!$node instanceof NodeInterface) {
      return [];
    }

    $node_id = $node->id();
    $uid = $node->getOwnerId();

    // Get data from config form.
    $config = $this->configFactory->get('related_content_block.settings');
    $taxonomy_fields = $config->get('related_taxonomy_fields') ?? [];
    $author_fields = $config->get('related_author_fields') ?? [];
    $enable_owner_filter = $config->get('enable_owner_filter');
    $limit = (int) $config->get('items_per_page') ?? 6;
    $selected_bundle = $config->get('selected_bundle') ?? 'articles';
    $selected_view_mode = $config->get('selected_view_mode') ?? 'teaser';
    $max_pages = (int) $config->get('limited_pages') ?? 3;
    $display_mode = $config->get('display') ?? 'one_column';

    // Get data from referenced fields.
    $tids = [];
    foreach ($taxonomy_fields as $field_name) {
      if ($node->hasField($field_name)) {
        foreach ($node->get($field_name)->getValue() as $term_ref) {
          $tids[] = $term_ref['target_id'];
        }
      }
    }

    $author_profile_ids = [];
    foreach ($author_fields as $field_name) {
      if ($node->hasField($field_name)) {
        foreach ($node->get($field_name)->getValue() as $author_ref) {
          $author_profile_ids[] = $author_ref['target_id'];
        }
      }
    }

    // Use the service to get related node IDs for the first page.
    $nids = $this->relatedContentService->getRelatedNodeIds(
      $node_id, $tids, $author_profile_ids, $uid, 0, $limit
    );

    $nodes = $this->entityTypeManager->getStorage('node')->loadMultiple($nids);
    $view_builder = $this->entityTypeManager->getViewBuilder('node');

    // Construct the display mode.
    $rendered_items = [];
    foreach ($nodes as $node) {
      $classes = ($display_mode === 'two_columns') ? ['col-md-12', 'col-lg-6'] : ['col-12'];

      $rendered_items[] = [
        '#type' => 'container',
        '#attributes' => ['class' => $classes],
        'content' => $view_builder->view($node, $selected_view_mode),
      ];
    }

    $build = [];
    $build['#attached']['drupalSettings']['RelatedContent'] = [
      'nid' => $node_id,
      'uid' => $uid,
      'tids' => $tids,
      'author_profile_ids' => $author_profile_ids,
      'limit' => $limit,
      'max_pages' => $max_pages,
    ];

    $build['related_items'] = [
      '#type' => 'container',
      '#attributes' => ['class' => ['row', 'gy-3', 'related-items']],
      'items' => $rendered_items,
    ];

    // Only add the "Load more" button if there are more pages to load.
    if ($max_pages > 1) {
      // AJAX Btn.
      $build['load_more'] = [
        '#type' => 'inline_template',
        '#template' => '
          <div class="d-flex justify-content-center mt-4">
            <button class="load-more btn btn-primary" data-page="1">
              <span class="button-text">{{ text }}</span>
              <span class="spinner-border spinner-border-sm ms-2 d-none spinner" role="status" aria-hidden="true"></span>
            </button>
          </div>',
        '#context' => [
          'text' => $this->t('Load more'),
        ],
      ];
    }

    $build['#cache'] = [
      'contexts' => ['url.path', 'user.permissions'],
      'tags' => ['node_list', 'related_content_block_config', 'node:' . $node->id()],
      'max-age' => 10800,
    ];

    $build['#attached']['library'][] = 'related_content_block/load_more';

    return $build;

  }

}
