<?php

namespace Drupal\related_content;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\node\NodeInterface;

/**
 * Service for fetching related content.
 */
class RelatedContentService {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Constructs a RelatedContentService object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, ConfigFactoryInterface $configFactory) {
    $this->entityTypeManager = $entityTypeManager;
    $this->configFactory = $configFactory;
  }

  /**
   * Gets related node IDs based on the given context.
   *
   * @param int $nid
   *   The node ID to find related content for.
   * @param array $tids
   *   An array of taxonomy term IDs.
   * @param array $author_profile_ids
   *   An array of author profile IDs.
   * @param int|null $uid
   *   The user ID of the node owner.
   * @param int $page
   *   The page number for pagination.
   * @param int $limit
   *   The number of items per page.
   *
   * @return int[]
   *   An array of related node IDs.
   */
  public function getRelatedNodeIds(int $nid, array $tids, array $author_profile_ids, ?int $uid, int $page, int $limit): array {
    $config = $this->configFactory->get('related_content.settings');
    $taxonomy_fields = $config->get('related_taxonomy_fields') ?? [];
    $author_fields = $config->get('related_author_fields') ?? [];
    $enable_owner_filter = $config->get('enable_owner_filter');
    $selected_bundle = $config->get('selected_bundle') ?? 'articles';

    $query = $this->entityTypeManager->getStorage('node')->getQuery();
    $query->accessCheck(TRUE);
    $query->condition('status', NodeInterface::PUBLISHED);
    $query->condition('type', $selected_bundle);
    $query->condition('nid', $nid, '<>');

    $group = $query->orConditionGroup();

    if (!empty($tids)) {
      foreach ($taxonomy_fields as $field_name) {
        if (!empty($field_name)) {
          $group->condition($field_name . '.target_id', $tids, 'IN');
        }
      }
    }

    if (!empty($author_profile_ids)) {
      foreach ($author_fields as $field_name) {
        if (!empty($field_name)) {
          $group->condition($field_name . '.target_id', $author_profile_ids, 'IN');
        }
      }
    }

    if ($enable_owner_filter && $uid) {
      $group->condition('uid', $uid);
    }

    $query->condition($group);
    $query->range($page * $limit, $limit);
    $query->sort('created', 'DESC');

    return $query->execute();
  }

}