<?php

namespace Drupal\related_content\Controller;

use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Drupal\Core\Controller\ControllerBase;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\related_content\RelatedContentService;

/**
 * Load More Controller.
 */
class LoadMoreController extends ControllerBase {

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The related content service.
   *
   * @var \Drupal\related_content\RelatedContentService
   */
  protected $relatedContentService;

  /**
   * Constructs a LoadMoreController.
   */
  public function __construct(EntityTypeManagerInterface $entityTypeManager, RendererInterface $renderer, RelatedContentService $relatedContentService) {
    $this->entityTypeManager = $entityTypeManager;
    $this->renderer = $renderer;
    $this->relatedContentService = $relatedContentService;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('renderer'),
      $container->get('related_content.service')
    );
  }

  /**
   * AJAX callback controller for loading more related content.
   *
   * This method handles requests to dynamically load additional related nodes
   * based on the current configuration and context (e.g., content type,
   * view mode, taxonomy/profile filters, and pagination).
   */
  public function load(Request $request): JsonResponse {
    $module_config = $this->config('related_content.settings');
    $nid = $request->query->get('nid');
    $page = (int) $request->query->get('page', 0);
    
    $default_limit = (int) $module_config->get('items_per_page') ?? 6;
    $max_pages = $module_config->get('limited_pages');

    // If we have already exceeded the limit, we do not return any more pages.
    if ($page >= $max_pages) {
      return new JsonResponse([
        'content' => '',
        'hasMore' => FALSE,
      ]);
    }

    // Request parameters.
    // Taxonomy terms of the current node.
    $tids = $request->query->all('tids');
    // Author profile.
    $author_profile_ids = $request->query->all('author_profile_ids');
    // Owner of the current node.
    $uid = $request->query->get('uid');
    $limit = (int) $request->query->get('limit', $default_limit);

    // Use the service to get related node IDs.
    $nids = $this->relatedContentService->getRelatedNodeIds(
      $nid, $tids, $author_profile_ids, $uid, $page, $limit
    );

    // The `hasMore` logic now correctly compares the next page with max_pages.
    $hasMore = !empty($nids) && ($page + 1) < $max_pages;

    $nodes = $this->entityTypeManager->getStorage('node')->loadMultiple($nids);
    $view_builder = $this->entityTypeManager->getViewBuilder('node');
    $selected_view_mode = $module_config->get('selected_view_mode') ?? 'teaser';
    $display_mode = $module_config->get('display') ?? 'one_column';

    // Construct the display mode.
    $rendered_items = [];
    foreach ($nodes as $node) {
      $classes = ($display_mode === 'two_columns') ? ['col-md-12', 'col-lg-6'] : ['col-12'];

      $rendered_items[] = [
        '#type' => 'container',
        '#attributes' => ['class' => $classes],
        'content' => $view_builder->view($node, $selected_view_mode),
      ];
    }

    // Inject rendered items.
    $build = ['items' => $rendered_items];
    $html = $this->renderer->renderRoot($build);

    return new JsonResponse([
      'content' => $html,
      'hasMore' => $hasMore,
    ]);
  }

}
