<?php

namespace Drupal\remote_file_importer_sftp\Plugin\RemoteFileImporter\Datasource;

use Drupal\Core\Form\FormStateInterface;
use Drupal\remote_file_importer\DataSourcePluginBase;
use Drupal\remote_file_importer\Entity\DataSource;
use phpseclib3\Net\SFTP;

/**
 * Provids SFTP import functionality.
 *
 * @RemoteFileImporterDataSourcePlugin(
 *   id = "rif_sftp_datasource",
 *   label = @Translation("SFTP"),
 * )
 */
class SftpDataSource extends DataSourcePluginBase {

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm($values = []) {

    $form = [];
    $form['host'] = [
      '#type' => 'textfield',
      '#default_value' => $values['host'] ?? '',
      '#title' => $this->t('Host'),
      '#description' => $this->t('Exclude protocol.'),
      '#required' => TRUE,
    ];

    $form['port'] = [
      '#type' => 'textfield',
      '#default_value' => $values['port'] ?? '',
      '#title' => $this->t('Port'),
      '#placeholder' => '21',
    ];

    $form['user'] = [
      '#type' => 'textfield',
      '#default_value' => $values['user'] ?? '',
      '#title' => $this->t('User'),
      '#required' => TRUE,
    ];

    $new = empty($values['host']);
    $form['change_pass'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Change password'),
      '#default_value' => $new,
      '#access' => !$new,
      '#description' => $this->t('Your password is stored; select to change it.'),
    ];

    $form['password'] = [
      '#type' => 'password',
      '#default_value' => $values['password'] ?? '',
      '#title' => $this->t('Password'),
      '#required' => $new,
      '#states' => [
        'visible' => [
          ':input[name="change_pass"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['remote_dir'] = [
      '#type' => 'textfield',
      '#default_value' => $values['remote_dir'] ?? '',
      '#title' => $this->t('Remote Dir'),
    ];

    return $form;

  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $input = $form_state->getValue('host');
    if ($input) {
      if (strpos($input, ':') !== FALSE || strpos($input, '//') !== FALSE) {
        $form_state->setErrorByName('host', $this->t('The host must not contain the protocol (e.g. sftp://).'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function establishConnection($settings) {

    $port = !empty($settings['port']) ? $settings['port'] : 22;

    // Set up a connection.
    $this->connection = new SFTP($settings['host'], $port);
    if ($this->connection->login($settings['user'], $settings['password'])) {
      if ($settings['remote_dir']) {
        $this->connection->chdir($settings['remote_dir']);
      }
      return TRUE;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function collectFiles() {
    $remoteDirectory = $this->connection->pwd();
    $files = $this->ftpGetAllFiles($remoteDirectory);
    return $files;
  }

  /**
   * Recursively get all files in a directory.
   *
   * @param string $dir
   *   The directory to search.
   *
   * @return array
   *   An array of file paths.
   */
  private function ftpGetAllFiles($dir) {
    $files = [];

    // Normalize directory path.
    $dir = rtrim($dir, '/');

    // Use recursive rawlist to get all files in directory and subdirectories.
    $rawlist = $this->connection->rawlist($dir, TRUE);

    if (!$rawlist) {
      return $files;
    }

    foreach ($rawlist as $fullPath => $file_info) {
      // Skip current and parent directory entries.
      $filename = basename($fullPath);
      if ($filename === '.' || $filename === '..') {
        continue;
      }

      // Check file type from rawlist data first (more efficient).
      if (isset($file_info->type) && $file_info->type === NET_SFTP_TYPE_REGULAR) {
        $files[] = [
          'path' => $fullPath,
          'metadata' => [
            'modified_date' => $file_info->mtime,
          ],
        ];
      }
      elseif (!isset($file_info->type) || $file_info->type !== NET_SFTP_TYPE_DIRECTORY) {
        // Fallback check for files only (skip directories in recursive mode).
        // The`is_file() function will re-init an SFTP connection on each call.
        if ($this->connection->is_file($fullPath)) {
          $files[] = [
            'path' => $fullPath,
            'metadata' => [
              'modified_date' => $file_info->mtime,
            ],
          ];
        }
      }
    }

    return $files;
  }

  /**
   * {@inheritdoc}
   */
  public function importFile($localFileName, $remoteFileName): bool {
    // Normalize path more efficiently.
    $remoteFilePath = str_replace('//', '/', $remoteFileName);
    return (bool) $this->connection->get($remoteFilePath, $localFileName);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteRemoteFile($remoteFileName): bool {
    // Normalize path more efficiently.
    $remoteFilePath = str_replace('//', '/', $remoteFileName);
    // Non-recursive deletion.
    return $this->connection->delete($remoteFilePath, FALSE);
  }

  /**
   * {@inheritdoc}
   */
  public function closeConnection() {
    $this->connection = NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function processConfigurationForm(array &$form, FormStateInterface $form_state, DataSource $entity) {
    $settings = [];
    $settingKeys = array_keys($this->buildConfigurationForm());
    foreach ($settingKeys as $key) {
      $settings[$key] = $form_state->getValue($key);
    }

    if (!$form_state->getValue('change_pass') && $entity->get('settings')) {
      $settings['password'] = $entity->get('settings')['password'];
    }

    if (array_key_exists('change_pass', $settings)) {
      unset($settings['change_pass']);
    }

    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultDateFormat() {
    return 'U';
  }

}
