<?php

namespace Drupal\remote_stream_wrapper\StreamWrapper;

use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\StreamWrapper\StreamWrapperInterface;
use Drupal\Core\Utility\Error;
use Drupal\remote_stream_wrapper\HttpClientTrait;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;

// phpcs:disable Drupal.NamingConventions.ValidFunctionName.ScopeNotCamelCaps
// Avoiding camel case warning for the stream wrapper methods: stream_* etc.
/**
 * HTTP(s) stream wrapper.
 */
class HttpStreamWrapper implements RemoteStreamWrapperInterface {

  use ReadOnlyPhpStreamWrapperTrait;
  use HttpClientTrait;

  /**
   * Stream context resource.
   *
   * @var resource
   */
  public $context;

  /**
   * The URI of the resource.
   *
   * @var string
   */
  protected string $uri;

  /**
   * The response stream.
   *
   * @var \Psr\Http\Message\StreamInterface
   */
  protected StreamInterface $stream;

  /**
   * Optional configuration for HTTP requests.
   *
   * @var array
   */
  protected array $config = [];

  /**
   * The Logger.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * {@inheritdoc}
   */
  public static function getType(): int {
    return StreamWrapperInterface::READ & StreamWrapperInterface::HIDDEN;
  }

  /**
   * {@inheritdoc}
   */
  public function getName(): string {
    return 'HTTP stream wrapper';
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string {
    return 'HTTP stream wrapper';
  }

  /**
   * {@inheritdoc}
   */
  public function setUri($uri): void {
    $this->uri = $uri;
  }

  /**
   * {@inheritdoc}
   */
  public function getUri(): string {
    return $this->uri;
  }

  /**
   * {@inheritdoc}
   */
  public function getExternalUrl(): string {
    return $this->uri;
  }

  /**
   * {@inheritdoc}
   */
  public function realpath(): string|bool {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function dirname($uri = NULL): string {
    if (!isset($uri)) {
      $uri = $this->uri;
    }

    [$scheme, $target] = explode('://', $uri, 2);
    $dirname = dirname($target);

    if ($dirname == '.') {
      $dirname = '';
    }

    return $scheme . '://' . $dirname;
  }

  /**
   * {@inheritdoc}
   *
   * @codeCoverageIgnore
   */
  public function stream_close() {
    // Nothing to do when closing an HTTP stream.
  }

  /**
   * {@inheritdoc}
   */
  public function stream_eof() {
    return $this->stream->eof();
  }

  /**
   * {@inheritdoc}
   */
  public function stream_lock($operation) {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_open($path, $mode, $options, &$opened_path) {
    if (!in_array($mode, ['r', 'rb', 'rt'])) {
      if ($options & STREAM_REPORT_ERRORS) {
        trigger_error('stream_open() write modes not supported for HTTP stream wrappers', E_USER_WARNING);
      }
      return FALSE;
    }

    try {
      $this->setUri($path);
      $this->request();
    }
    catch (\Exception $e) {
      if ($options & STREAM_REPORT_ERRORS) {
        // @todo Make this testable.
        Error::logException($this->getLogger(), $e);
      }
      return FALSE;
    }

    if ($options & STREAM_USE_PATH) {
      $opened_path = $path;
    }

    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_read($count) {
    return $this->stream->read($count);
  }

  /**
   * {@inheritdoc}
   */
  public function stream_seek($offset, $whence = SEEK_SET) {
    try {
      $this->stream->seek($offset, $whence);
    }
    catch (\RuntimeException $e) {
      // @todo Make this testable.
      Error::logException($this->getLogger(), $e);
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Change stream options.
   *
   * This method is called to set options on the stream.
   *
   * @param int $option
   *   One of:
   *   - STREAM_OPTION_BLOCKING: The method was called in response to
   *     stream_set_blocking().
   *   - STREAM_OPTION_READ_TIMEOUT: The method was called in response to
   *     stream_set_timeout().
   *   - STREAM_OPTION_WRITE_BUFFER: The method was called in response to
   *     stream_set_write_buffer().
   * @param int $arg1
   *   If option is:
   *   - STREAM_OPTION_BLOCKING: The requested blocking mode:
   *     - 1 means blocking.
   *     - 0 means not blocking.
   *   - STREAM_OPTION_READ_TIMEOUT: The timeout in seconds.
   *   - STREAM_OPTION_WRITE_BUFFER: The buffer mode, STREAM_BUFFER_NONE or
   *     STREAM_BUFFER_FULL.
   * @param int $arg2
   *   If option is:
   *   - STREAM_OPTION_BLOCKING: This option is not set.
   *   - STREAM_OPTION_READ_TIMEOUT: The timeout in microseconds.
   *   - STREAM_OPTION_WRITE_BUFFER: The requested buffer size.
   *
   * @return bool
   *   TRUE on success, FALSE otherwise. If $option is not implemented, FALSE
   *   should be returned.
   */
  public function stream_set_option($option, $arg1, $arg2) {
    if ($option != STREAM_OPTION_READ_TIMEOUT) {
      return FALSE;
    }

    $this->config['timeout'] = $arg1 + ($arg2 / 1000000);
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function stream_stat() {
    // @see https://github.com/guzzle/psr7/blob/master/src/StreamWrapper.php
    $stat = [
      // Device number.
      'dev' => 0,
      // Inode number.
      'ino' => 0,
      // Inode protection (regular file + read only)
      'mode' => 0100000 | 0444,
      // Number of links.
      'nlink' => 0,
      // Userid of owner.
      'uid' => 0,
      // Groupid of owner.
      'gid' => 0,
      // Device type, if inode device *.
      'rdev' => 0,
      // Size in bytes.
      'size' => 0,
      // Time of last access (Unix timestamp)
      'atime' => 0,
      // Time of last modification (Unix timestamp)
      'mtime' => 0,
      // Time of last inode change (Unix timestamp)
      'ctime' => 0,
      // Blocksize of filesystem IO.
      'blksize' => 0,
      // Number of blocks allocated.
      'blocks' => 0,
    ];

    try {
      $response = $this->requestTryHeadLookingForHeader($this->uri, 'Content-Length', $this->config);

      if ($response->hasHeader('Content-Length')) {
        $stat['size'] = (int) $response->getHeaderLine('Content-Length');
      }
      elseif ($size = $response->getBody()->getSize()) {
        $stat['size'] = $size;
      }
      if ($response->hasHeader('Last-Modified')) {
        if ($mtime = strtotime($response->getHeaderLine('Last-Modified'))) {
          $stat['mtime'] = $mtime;
        }
      }

      return $stat;
    }
    catch (\Exception $e) {
      Error::logException($this->getLogger(), $e);
      return FALSE;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function stream_tell() {
    return $this->stream->tell();
  }

  /**
   * {@inheritdoc}
   */
  public function url_stat($path, $flags) {
    $this->setUri($path);
    if ($flags & STREAM_URL_STAT_QUIET) {
      return @$this->stream_stat();
    }
    else {
      return $this->stream_stat();
    }
  }

  /**
   * Returns the current HTTP client configuration.
   *
   * @return array
   *   The HTTP client configuration.
   */
  public function getHttpConfig() {
    return $this->config;
  }

  /**
   * {@inheritdoc}
   */
  public function request(string $method = 'GET'): ResponseInterface {
    $response = $this->getHttpClient()
      ->request($method, $this->uri, $this->config);
    if ($method !== 'HEAD') {
      $this->stream = $response->getBody();
    }
    return $response;
  }

  /**
   * Sets the logger channel.
   *
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   A logger channel.
   */
  public function setLogger(LoggerChannelInterface $logger): void {
    $this->logger = $logger;
  }

  /**
   * Returns the logger channel.
   *
   * @return \Drupal\Core\Logger\LoggerChannelInterface
   *   The logger channel.
   */
  public function getLogger(): LoggerChannelInterface {
    if (!isset($this->logger)) {
      $this->logger = \Drupal::logger('remote_stream_wrapper');
    }

    return $this->logger;
  }

}

// phpcs:enable
