<?php

namespace Drupal\renderable_options\Hook;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;

/**
 * Implements preprocess hooks.
 */
class PreprocessHooks {

  #[Hook('theme')]
  /**
   * Implements hook_theme().
   */
  public function theme(array $existing, string $type, string $theme, string $path): array {
    return [
      'form_element_label_rich_wrapper' => [
        'render element' => 'element',
      ],
    ];
  }

  #[Hook('preprocess_form_element')]
  /**
   * Implements hook_preprocess_HOOK().
   */
  public function preprocessFormElement(array &$variables): void {
    $element = $variables['element'];
    if (!empty($element['#title_renderable'])) {
      $variables['label']['#title_renderable'] = $element['#title_renderable'];
      $variables['label']['#theme_wrappers'][] = 'form_element_label_rich_wrapper';
      $variables['#attached']['library'][] = 'renderable_options/widget';
    }
  }

  #[Hook('preprocess_form_element_label_rich_wrapper')]
  /**
   * Implements hook_preprocess_HOOK().
   */
  public function preprocessFormElementLabelRichWrapper(&$variables): void {
    $element = $variables['element'];
    $variables['attributes']['class'][] = 'form-element-label-rich-wrapper';
    $variables['content'] = $element['#title_renderable'];
    $variables['label'] = $element['#children'];
  }

  #[Hook('element_info_alter')]
  /**
   * Implements hook_element_info_alter().
   */
  public function elementInfoAlter(array &$info): void {
    $info['checkboxes']['#process'][] = [static::class, 'processOptions'];
    $info['checkbox']['#process'][] = [static::class, 'processOptions'];
    $info['radios']['#process'][] = [static::class, 'processOptions'];
    $info['select']['#process'][] = [static::class, 'processOptions'];
  }

  /**
   * Process renderable options.
   *
   * @param array $element
   *   The element.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $complete_form
   *   The complete form.
   *
   * @return array
   *   The element.
   */
  public static function processOptions(array &$element, FormStateInterface $form_state, array &$complete_form): array {
    if (empty($element['#options'])) {
      return $element;
    }

    foreach ($element['#options'] as $key => $option) {
      if (!is_object($option)) {
        continue;
      }

      if (!isset($option->option)) {
        continue;
      }

      $element['#options'][$key] = $option->option;
      $element['#options_renderable'][$key] = $option->renderable ?? NULL;

      if (isset($element[$key])) {
        $element[$key]['#title'] = $element['#options'][$key];
        $element[$key]['#title_renderable'] = $element['#options_renderable'][$key];
      }
    }
    return $element;
  }

}
