<?php

namespace Drupal\replicate_actions\EventSubscriber;

use Drupal\Core\Session\AccountInterface;
use Drupal\replicate\Events\ReplicateAlterEvent;
use Drupal\replicate\Events\ReplicatorEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Sets owner of replicated entities to the current user.
 */
class ReplicateSetAuthor implements EventSubscriberInterface {

  /**
   * @var \Drupal\Core\Session\AccountInterface
   */
  private AccountInterface $currentUser;

  /**
   * ReplicateSetAuthor constructor.
   *
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *    The current user.
   */
  public function __construct(AccountInterface $currentUser) {
    $this->currentUser = $currentUser;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    $events = [];
    $events[ReplicatorEvents::REPLICATE_ALTER][] = ['setAuthor', 1];

    return $events;
  }

  /**
   * Sets the author of a replicated node to the current user.
   *
   * @param \Drupal\replicate\Events\ReplicateAlterEvent $event
   *   The event fired by the replicator.
   */
  public function setAuthor(ReplicateAlterEvent $event): void {
    $clonedEntity = $event->getEntity();

    // We don't need to check if it's instanceof TranslatableInterface since
    // at this point it's guaranteed to be a content entity (which implement
    // TranslatableInterface). If the above check changes in the future,
    // we would need to make sure this is translatable before getting
    // languages.
    foreach ($clonedEntity->getTranslationLanguages() as $translation_language) {
      /**
       * @var \Drupal\Core\Entity\ContentEntityTypeInterface|\Drupal\Core\Entity\EntityPublishedInterface $translation
       */
      $translation = $clonedEntity->getTranslation($translation_language->getId());

      // Set the author to the current user.
      $uid = $this->currentUser->id();
      if (method_exists($clonedEntity, 'setOwnerId')) {
        $translation->setOwnerId($uid);
      }

    }
  }

}
