<?php

namespace Drupal\Tests\replicate_actions\Functional;

use Drupal\Core\Cache\Cache;
use Drupal\node\Entity\NodeType;
use Drupal\node\NodeInterface;
use Drupal\Tests\BrowserTestBase;
use Drupal\user\UserInterface;

/**
 * Tests the UI functionality.
 *
 * @group replicate
 */
class ReplicateActionsTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'replicate',
    'replicate_ui',
    'replicate_actions',
    'replicate_actions_test',
    'node',
    'block',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The user's object.
   *
   * @var \Drupal\user\UserInterface
   */
  protected UserInterface $user;

  /**
   * The node's object.
   *
   * @var \Drupal\node\NodeInterface
   */
  protected NodeInterface $node;

  /**
   * Initial setup for testing.
   *
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->user = $this->drupalCreateUser([
      'bypass node access',
      'administer nodes',
      'replicate entities',
    ]);
    $node_type = NodeType::create([
      'type' => 'page',
    ]);
    $node_type->save();

    $this->placeBlock('local_tasks_block');
    $this->placeBlock('system_messages_block');
    \Drupal::configFactory()
      ->getEditable('replicate_ui.settings')
      ->set('entity_types', ['node'])
      ->save();
    \Drupal::service('router.builder')->rebuild();
    Cache::invalidateTags(['entity_types']);
  }

  /**
   * Test redirect and the node's status.
   */
  public function testFunctionality() {
    $node = $this->drupalCreateNode([
      'title' => $this->randomMachineName(8),
      'type' => 'page',
    ]);

    $this->drupalLogin($this->user);
    $node_storage = $this->container->get('entity_type.manager')
      ->getStorage('node');

    // Verify the node published.
    $this->assertTrue($node->isPublished(), 'Node is published now.');

    $this->drupalGet($node->toUrl());

    $this->getSession()->getPage()->clickLink('Replicate');
    $this->assertEquals(200, $this->getSession()
      ->getDriver()
      ->getStatusCode());
    $this->getSession()->getPage()->pressButton('Replicate');

    // Verify the new replicated node is unpublished.
    $replicated_node = $node_storage->load($node->id() + 1);
    $this->assertFalse($replicated_node->isPublished(), 'Node is unpublished now.');
    $this->assertStringEndsWith('(Copy)', $replicated_node->label());

    // Verify the user was redirected to /node/*/edit.
    $this->assertStringEndsWith($replicated_node->toUrl('edit-form')
      ->toString(), $this->getSession()->getCurrentUrl());
  }

  /**
   * Test if custom submit handlers for the replicate form can finish.
   */
  public function testLongRunningCustomFormSubmitHandlers() {
    // Adding '[x]' to the node title will trigger the custom submit handler in
    // replicate_actions_test.module.
    $node = $this->drupalCreateNode([
      'title' => $this->randomMachineName(8) . ' [x]',
      'type' => 'page',
    ]);

    $this->drupalLogin($this->user);
    $node_storage = $this->container->get('entity_type.manager')
      ->getStorage('node');

    $this->drupalGet($node->toUrl());

    $this->getSession()->getPage()->clickLink('Replicate');
    $this->getSession()->getPage()->pressButton('Replicate');
    $replicated_node = $node_storage->load($node->id() + 1);
    $this->assertStringEndsWith('[25] (Copy)', $replicated_node->label());

    $this->assertStringEndsWith($replicated_node->toUrl('edit-form')
      ->toString(), $this->getSession()->getCurrentUrl());
  }

}
