<?php

namespace Drupal\reporting\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\PagerSelectExtender;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Symfony\Component\HttpFoundation\Request;

/**
 * Show violation reports from the DbLog.
 */
class DbLogController extends ControllerBase {

  /**
   * The database service.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $database;

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * Constructs a DbLogController object.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   A database connection.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   A module handler.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   */
  public function __construct(
    Connection $database,
    ModuleHandlerInterface $module_handler,
    DateFormatterInterface $date_formatter,
  ) {
    $this->database = $database;
    $this->moduleHandler = $module_handler;
    $this->dateFormatter = $date_formatter;
  }

  /**
   * Displays a listing of reports.
   *
   * Full-length messages can be viewed on the report details page.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return array
   *   A render array as expected by
   *   \Drupal\Core\Render\RendererInterface::render().
   */
  public function overview(Request $request) {
    $rows = [];

    $header = [
      [
        'data' => $this->t('Endpoint'),
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
      [
        'data' => $this->t('Date'),
        'class' => [RESPONSIVE_PRIORITY_MEDIUM],
      ],
      [
        'data' => $this->t('Type'),
      ],
      [
        'data' => $this->t('Disposition'),
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
      [
        'data' => $this->t('Location'),
        'class' => [RESPONSIVE_PRIORITY_LOW],
      ],
      $this->t('Operations'),
    ];

    $query = $this->database->select('watchdog', 'w')
      ->fields('w', [
        'wid',
        'timestamp',
        'variables',
      ])
      ->condition('type', 'reporting')
      ->condition('message', '@endpoint <br/>%', 'LIKE')
      ->orderBy('timestamp', 'DESC')
      ->extend(PagerSelectExtender::class)
      ->limit(50);

    $result = $query->execute();

    foreach ($result as $dblog) {
      $variables = @unserialize($dblog->variables);

      $endpoint = $variables['@endpoint'];
      $data = json_decode($variables['@data']);

      $rows[] = [
        'data' => [
          $endpoint,
          $this->dateFormatter->format($dblog->timestamp, 'short'),
          $data->type,
          $data->body->disposition,
          $data->url,
          Link::fromTextAndUrl(
            $this->t('View'),
            new Url('dblog.event', ['event_id' => $dblog->wid])
          ),
        ],
      ];
    }

    $build['dblog_table'] = [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No violation reports available.'),
    ];
    $build['dblog_pager'] = ['#type' => 'pager'];

    return $build;
  }

}
