<?php

namespace Drupal\reporting\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\reporting\Entity\ReportingEndpointInterface;
use Drupal\reporting\ReportingResponse;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Controller for receiving violation reports.
 */
class ReportingEndpoint extends ControllerBase {

  /**
   * The Request Stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  private $requestStack;

  /**
   * The Logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  private $logger;

  /**
   * Create a new Report URI Controller.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The Request Stack service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The Logger channel.
   */
  public function __construct(RequestStack $requestStack, LoggerInterface $logger) {
    $this->requestStack = $requestStack;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack'),
      $container->get('logger.factory')->get('reporting')
    );
  }

  /**
   * Handle a report submission.
   *
   * @param \Drupal\reporting\Entity\ReportingEndpointInterface $reporting_endpoint
   *   The reporting endpoint.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   A Response object.
   */
  public function log(ReportingEndpointInterface $reporting_endpoint) {

    // Return 410: Gone if endpoint is disabled.
    // @see https://w3c.github.io/reporting/#try-delivery
    if (!$reporting_endpoint->status()) {
      return new ReportingResponse(410);
    }

    $request = $this->requestStack->getCurrentRequest();

    // Return 405: Method Not Allowed if not a POST request.
    // This is used instead of the 'methods' property on the route so that an
    // empty response body can be returned instead of a rendered error page.
    if ($request->getMethod() !== 'POST') {
      return new ReportingResponse(405);
    }

    $report = json_decode($request->getContent(), TRUE);

    // Return 400: Bad Request if content cannot be parsed.
    if (empty($report) || json_last_error() != JSON_ERROR_NONE) {
      return new ReportingResponse(400);
    }

    switch ($request->headers->get('Content-Type')) {
      case 'application/reports+json':
        $this->storeReportToData($reporting_endpoint, $report, $request);
        break;

      case 'application/csp-report':
        $this->storeReportUriData($reporting_endpoint, $report, $request);
        break;

      default:
        // 415: Unsupported Media Type.
        return new ReportingResponse(415);
    }

    // 202: Accepted.
    return new ReportingResponse(202);
  }

  /**
   * Helper to log CSP report sent via report-uri directive.
   *
   * Firefox sends data in a unique format that needs conversion.
   * Safari sends single reports in the same format as report-to.
   *
   * @param \Drupal\reporting\Entity\ReportingEndpointInterface $reporting_endpoint
   *   The reporting endpoint.
   * @param array<string, mixed> $report
   *   A CSP violation report.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The Request object.
   */
  private function storeReportUriData(ReportingEndpointInterface $reporting_endpoint, array $report, Request $request): void {
    // Convert data from Firefox to format expected by Reporting API.
    if (array_key_exists('csp-report', $report)) {
      $report = [
        'type' => 'csp-violation',
        'url' => $report['csp-report']['document-uri'],
        'body' => $report['csp-report'],
      ];

      $keyMap = [
        "document-uri" => "documentURL",
        "blocked-uri" => "blockedURL",
        "effective-directive" => "effectiveDirective",
        "violated-directive" => "violatedDirective",
        "original-policy" => "originalPolicy",
        "source-file" => "sourceFile",
        "script-sample" => "sample",
        "status-code" => "statusCode",
        "column-number" => "columnNumber",
        "line-number" => "lineNumber",
      ];

      foreach (array_intersect_key($keyMap, $report['body']) as $originalKey => $newKey) {
        $report['body'][$newKey] = $report['body'][$originalKey];
        unset($report['body'][$originalKey]);
      }
      ksort($report['body']);
    }

    $this->storeReportToData($reporting_endpoint, [$report], $request);
  }

  /**
   * Helper to log reports sent to Report-To endpoint.
   *
   * @param \Drupal\reporting\Entity\ReportingEndpointInterface $reporting_endpoint
   *   The reporting endpoint.
   * @param array<array<string, mixed>> $reports
   *   An array of violation reports.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The Request object.
   */
  private function storeReportToData(ReportingEndpointInterface $reporting_endpoint, array $reports, Request $request): void {
    foreach ($reports as $report) {
      // Ignore reports without expected format or data.
      if (
        !array_key_exists('type', $report)
        || !array_key_exists('url', $report)
        || !array_key_exists('body', $report)
      ) {
        continue;
      }

      if (!array_key_exists('user_agent', $report)) {
        $report['user_agent'] = $request->headers->get('User-Agent');
      }
      if (!array_key_exists('age', $report)) {
        $report['age'] = 0;
      }

      $this->logger
        ->info("@endpoint <br/>\n<pre>@data</pre>", [
          '@endpoint' => $reporting_endpoint->id(),
          '@data' => json_encode($report, JSON_PRETTY_PRINT),
        ]);
    }
  }

}
