<?php

namespace Drupal\representative_image\Plugin\migrate\source;

use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\State\StateInterface;
use Drupal\migrate\Attribute\MigrateSource;
use Drupal\migrate\Plugin\MigrationInterface;
use Drupal\migrate\Row;
use Drupal\migrate_drupal\Plugin\migrate\source\DrupalSqlBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Fetches representative image fields from the source database.
 */
#[MigrateSource('d7_representative_image_field_config')]
class FieldConfig extends FieldStorageConfig implements ContainerFactoryPluginInterface {

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected EntityTypeBundleInfoInterface $entityTypeBundleInfo;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, MigrationInterface $migration, StateInterface $state, EntityTypeManagerInterface $entity_type_manager, EntityTypeBundleInfoInterface $entity_type_bundle_info) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $migration, $state, $entity_type_manager);
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition, ?MigrationInterface $migration = NULL) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $migration,
      $container->get('state'),
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function prepareRow(Row $row) {
    $entity_definitions = $this->entityTypeManager->getDefinitions();
    $name = $row->getSourceProperty('name');
    // Variables are made of representative_image_field_[entity type]_[bundle].
    // First let's find a matching entity type from the variable name.
    foreach ($entity_definitions as $entity_type => $definition) {
      if (strpos($name, 'representative_image_field_' . $entity_type . '_') === 0) {
        // Extract the bundle out of the variable name.
        preg_match('/^representative_image_field_' . $entity_type . '_([a-zA-z0-9_]+)$/', $name, $matches);
        $bundle = $matches[1];

        // Check that the bundle exists.
        $bundles = $this->entityTypeBundleInfo->getBundleInfo($entity_type);
        if (!in_array($bundle, array_keys($bundles))) {
          // No matching bundle found in destination.
          return FALSE;
        }

        $row->setSourceProperty('entity_type', $entity_type);
        $row->setSourceProperty('bundle', $bundle);
        $row->setSourceProperty('settings', [
          'representative_image_field_name' => unserialize($row->getSourceProperty('value')),
        ]);
        return DrupalSqlBase::prepareRow($row);
      }
    }

    // No matching entity type found in destination for this variable.
    return FALSE;
  }

}
