<?php

namespace Drupal\Tests\representative_image\FunctionalJavascript;

use Drupal\FunctionalJavascriptTests\DrupalSelenium2Driver;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;
use Drupal\Tests\image\Kernel\ImageFieldCreationTrait;
use Drupal\Tests\TestFileCreationTrait;

/**
 * Contains common logic for functional tests.
 */
abstract class RepresentativeImageTestBase extends WebDriverTestBase {

  use ImageFieldCreationTrait;
  use TestFileCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystem
   */
  protected $fileSystem;

  /**
   * The node storage service.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $nodeStorage;

  /**
   * The representative image picker service.
   *
   * @var \Drupal\representative_image\RepresentativeImagePicker
   */
  protected $representativeImagePicker;

  /**
   * The default image file being used by representative_image.
   *
   * @var \stdClass
   */
  protected $defaultImageFile;

  /**
   * {@inheritdoc}
   */
  protected $minkDefaultDriverClass = DrupalSelenium2Driver::class;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'block',
    'field_ui',
    'file',
    'image',
    'node',
    'representative_image',
    'system',
    'user',
    'views_ui',
  ];

  /**
   * The admin user account.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->fileSystem = \Drupal::service('file_system');
    $this->nodeStorage = \Drupal::entityTypeManager()->getStorage('node');
    $this->representativeImagePicker = \Drupal::service('representative_image.picker');

    $this->adminUser = $this->drupalCreateUser([
      'access content',
      'access administration pages',
      'administer site configuration',
      'administer image styles',
      'administer nodes',
      'administer node display',
      'administer content types',
      'administer node fields',
      'bypass node access',
      'administer users',
      'administer user fields',
      'administer views',
    ]);
    $this->drupalLogin($this->adminUser);
    $this->drupalPlaceBlock('local_tasks_block');
    $this->setUpArticleContentType();
  }

  /**
   * Configures the article content type for testing representative image.
   *
   * Adds two image fields and a representative image field. Then hides
   * the image fields from the default display and shows the
   * representative image field.
   */
  protected function setUpArticleContentType() {
    // Create article content type with two image fields.
    $this->defaultImageFile = $this->randomFile('image');

    $this->drupalCreateContentType(['type' => 'article', 'name' => 'Article']);
    $this->createImageField('field_image1', 'node', 'article');
    $this->createImageField('field_image2', 'node', 'article');

    $page = $this->getSession()->getPage();
    $assert_session = $this->assertSession();
    $web_driver = $this->getSession()->getDriver();

    // Add representative image field with default image. Set it to field1.
    $this->drupalGet('/admin/structure/types/manage/article/fields/add-field');
    $this->clickLink('Representative Image');
    $this->assertSession()->assertWaitOnAjaxRequest();
    $page->fillField('label', 'Representative Image');
    $result = $this->assertSession()->waitForText('field_representative_image');
    $this->assertNotEmpty($result, "\"field_representative_image\" not found");
    $this->assertSession()->elementExists('xpath', '//button[text()="Continue"]')->press();
    $this->assertSession()->assertWaitOnAjaxRequest();

    $remote_path = $web_driver->uploadFileAndGetRemoteFilePath($this->fileSystem->realpath($this->defaultImageFile->uri));

    $file_field = $this->assertSession()->elementExists('xpath', '//input[@type="file"]');
    $file_field->setValue($remote_path);
    $this->assertSession()->waitForElementVisible('css', '[data-drupal-selector="edit-field-storage-subform-settings-default-image-uuid-file-2-filename"]');
    $page->find('css', '.ui-dialog-buttonset')->pressButton('Save');
    $this->assertTrue($assert_session->waitForText('Saved Representative Image configuration.'));

    // Adjust display settings so only Representative Image is shown.
    // Toggle row weight selects as visible.
    $this->drupalGet('/admin/structure/types/manage/article/display');
    $page->findButton('Show row weights')->click();
    $page->fillField('fields[field_representative_image][region]', 'content');
    $page->fillField('fields[field_image1][region]', 'hidden');
    $page->fillField('fields[field_image2][region]', 'hidden');
    $page->pressButton('Save');

    // @todo Need to do this or
    //   RepresentativeImagePicker::getRepresentativeImageField() will return
    //   ''.
    drupal_flush_all_caches();
  }

  /**
   * Correct field assertion for generated image.
   *
   * Asserts that an image generated by getTestFiles(image) was displayed
   * correctly.
   *
   * @param object $image
   *   An image object as returned by getTestFiles(image).
   * @param string $haystack
   *   (optional) A string to search for the image path. If provided preg_match
   *   will be used otherwise assertPattern will be called.
   * @param string $message
   *   (optional) The message to display with the test results.
   */
  public function assertImage($image, $haystack = '', $message = '') {
    $message = empty($message) ? $image->name . ' was correctly displayed' : $message;
    [$filename, $extension] = explode('.', $image->filename);
    $pattern = '/' . $filename . '(_?[0-9]?\.{1})' . $extension . '/';

    if (!empty($haystack)) {
      $this->assertTrue((bool) preg_match($pattern, $haystack), $message);
    }
    else {
      $this->assertSession()->responseMatches($pattern);
    }
  }

  /**
   * Provides a random image object.
   *
   * @param string $type
   *   The type of file to be created for testing purposes.
   *
   * @return object
   *   A file object which has the following attributes:
   *   - $file->url (for example, public://image-2.jpg)
   *   - $file->filename (for example, image-2.jpg)
   *   - $file->name (for example, image-2)
   */
  public function randomFile($type = 'image') {
    // Get all test images in the form of an array.
    /** @var \stdClass[] $files */
    $files = $this->getTestFiles($type);
    // Get the next one on the list, wrapping around if necessary.
    static $i = 0;
    return $files[($i++) % count($files)];
  }

}
