<?php

declare(strict_types=1);

namespace Drupal\request_logger_reports_test\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\page_cache\StackMiddleware\PageCache;
use Drupal\request_logger\StackMiddleware\RequestLoggerStackMiddleware;
use Drupal\request_logger_reports_test\TimeTesting;
use Drupal\test_helpers\TestHelpers;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Controller for the request_logger_reports_test test page.
 */
class RequestGenerator extends ControllerBase {

  /**
   * Constructs a RequestGenerator object.
   */
  public function __construct(private RequestLoggerStackMiddleware $service, private TimeTesting $timeService) {
  }

  /**
   * Creates a RequestGenerator instance.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get(RequestLoggerStackMiddleware::class),
      $container->get('datetime.time')
    );
  }

  /**
   * A cacheable page.
   */
  public function generateHitMissRequests() {
    $avgRequestPerHour = 150;
    $avgStep = 60 * 60 / $avgRequestPerHour;

    $logTimeMin = strtotime("00:00:00 -1 days");
    $logTimeMax = strtotime("23:59:59 -1 days");

    $dropCaches = [
      strtotime("09:55:23 -1 days"),
      strtotime("15:30:45 -1 days"),
    ];
    $dropCacheDelta = 0;

    $logTime = $logTimeMin;
    $cacheRateMax = 99;
    $cacheRate = $cacheRateMax;
    do {
      $step = rand((int) ($avgStep * 0.6), (int) ($avgStep * 1.4));
      $logTime += $step;
      $cacheRate += rand(0, 100) * 0.002;
      $cacheRate = min($cacheRateMax, $cacheRate);
      $node = rand(1, 300);

      if (
        isset($dropCaches[$dropCacheDelta])
        && $logTime > $dropCaches[$dropCacheDelta]
      ) {
        $cacheRate = 0;
        $dropCacheDelta++;
      }

      $request = Request::create("/node/$node");
      $pageCache = rand(0, 100) < $cacheRate ? 'HIT' : 'MISS';
      $response = new Response(
        headers: [
          PageCache::HEADER => $pageCache,
        ],
      );
      $output[] = 'Time ' . date('c', $logTime) . " - rate $cacheRate cache $pageCache";
      $this->timeService->setTime($logTime);
      TestHelpers::callPrivateMethod($this->service, 'logRequestAndResponse', [$request, $response]);
    } while ($logTime < $logTimeMax);

    return ['#markup' => implode('<br/>', $output), '#cache' => ['max-age' => 0]];
  }

}
