<?php

declare(strict_types=1);

namespace Drupal\request_logger_reports_test;

use Drupal\Component\Datetime\Time;

/**
 * Extends the Drupal Time service for obtaining custom system time for tests.
 */
class TimeTesting extends Time {

  /**
   * Custom timestamp for testing purposes.
   *
   * @var int|null
   */
  protected $customTime;

  /**
   * Custom microtime for testing purposes.
   *
   * @var float|null
   */
  protected $customMicroTime;

  /**
   * Custom request timestamp for testing purposes.
   *
   * @var int|null
   */
  protected $customRequestTime;

  /**
   * Custom request microtime for testing purposes.
   *
   * @var float|null
   */
  protected $customRequestMicroTime;

  /**
   * Sets a custom time for testing purposes.
   *
   * @param string $time
   *   A time string that can be parsed by strtotime().
   */
  public function setTime(string|int|float $time) {
    if (!is_numeric($time)) {
      $time = strtotime($time);
    }
    $this->customTime = $time;
    if (is_float($time)) {
      $timeFloat = $time;
    }
    else {
      // Use microtime(TRUE) to get a float and extract the fractional part so
      // we can add it to the integer timestamp parsed from the provided time
      // string. This avoids performing arithmetic on the string returned by
      // microtime(FALSE), which triggers "A non-numeric value encountered".
      $microNow = microtime(TRUE);
      $fraction = $microNow - floor($microNow);
      $timeFloat = $time + $fraction;
    }
    $this->customMicroTime = $timeFloat;
  }

  /**
   * Sets a custom request time for testing purposes.
   *
   * @param string $time
   *   A time string that can be parsed by strtotime().
   */
  public function setRequestTime(string $time) {
    $this->customRequestTime = strtotime($time);
    // Same approach as setTime() — keep the fractional microseconds and add
    // them to the integer timestamp for the requested time.
    $now = microtime(TRUE);
    $fraction = $now - floor($now);
    $this->customRequestMicroTime = strtotime($time) + $fraction;
  }

  /**
   * {@inheritdoc}
   */
  public function getRequestTime() {
    if (isset($this->customRequestTime)) {
      return $this->customRequestTime;
    }
    if (isset($this->customTime)) {
      return $this->customTime;
    }
    return parent::getRequestTime();
  }

  /**
   * {@inheritdoc}
   */
  public function getRequestMicroTime() {
    if (isset($this->customRequestMicroTime)) {
      return $this->customRequestMicroTime;
    }
    if (isset($this->customMicroTime)) {
      return $this->customMicroTime;
    }
    return parent::getRequestMicroTime();
  }

  /**
   * {@inheritdoc}
   */
  public function getCurrentTime() {
    if (isset($this->customTime)) {
      return $this->customTime;
    }
    return parent::getCurrentTime();
  }

  /**
   * {@inheritdoc}
   */
  public function getCurrentMicroTime() {
    if (isset($this->customMicroTime)) {
      return $this->customMicroTime;
    }
    return parent::getCurrentMicroTime();
  }

  /**
   * Returns a mimic of the timestamp of the current request.
   *
   * @return int
   *   A value returned by time().
   */
  protected function getProxyRequestTime(): int {
    if (isset($this->customRequestTime)) {
      return $this->customRequestTime;
    }
    if (isset($this->customTime)) {
      return $this->customTime;
    }
    return parent::getProxyRequestTime();
  }

  /**
   * Returns a mimic of the timestamp of the current request.
   *
   * @return float
   *   A value returned by microtime().
   */
  protected function getProxyRequestMicroTime(): float {
    if (isset($this->customRequestMicroTime)) {
      return $this->customRequestMicroTime;
    }
    if (isset($this->customMicroTime)) {
      return $this->customMicroTime;
    }
    return parent::getProxyRequestMicroTime();
  }

}
