<?php

namespace Drupal\request_logger\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\logger\Event\LoggerLogEvent;
use Drupal\request_logger\StackMiddleware\RequestLoggerStackMiddleware;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * The event that is triggered after a response is generated.
 *
 * @package Drupal\ai\EventSubscriber
 */
class LoggerLogEventSubscriber implements EventSubscriberInterface {

  /**
   * Config key to enable adding the request UUID to logs.
   */
  public const CONFIG_KEY_ADD_REQUEST_UUID = 'add_request_uuid_to_logs';

  /**
   * Constructor.
   */
  public function __construct(
    protected RequestStack $requestStack,
    protected ConfigFactoryInterface $configFactory,
  ) {
  }

  /**
   * {@inheritdoc}
   *
   * @return array
   *   The pre generate response event.
   */
  public static function getSubscribedEvents(): array {
    // @todo Subscribe only when the config is enabled.
    return [
      LoggerLogEvent::class => 'enhanceLogEntry',
    ];
  }

  /**
   * Enhance a LoggerLogEvent by adding request UUID metadata when enabled.
   *
   * @param \Drupal\logger\Event\LoggerLogEvent $event
   *   The logger event to enhance.
   */
  public function enhanceLogEntry(LoggerLogEvent $event): void {
    $config = $this->configFactory->get(RequestLoggerStackMiddleware::CONFIG_NAME);

    if (
      $config->get(self::CONFIG_KEY_ADD_REQUEST_UUID)
      && $request = $this->requestStack->getCurrentRequest()
    ) {
      $event->entry->set(RequestLoggerStackMiddleware::LOG_ENTRY_KEY_REQUEST_UUID, $request->attributes->get(RequestLoggerStackMiddleware::REQUEST_ATTRIBUTE_UUID));
      $event->entry->set(RequestLoggerStackMiddleware::LOG_ENTRY_KEY_MAIN_REQUEST_UUID, $request->attributes->get(RequestLoggerStackMiddleware::REQUEST_ATTRIBUTE_MAIN_UUID));
    }
  }

}
