<?php

namespace Drupal\request_logger\Form;

use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\RfcLogLevel;
use Drupal\request_logger\EventSubscriber\LoggerLogEventSubscriber;
use Drupal\request_logger\StackMiddleware\RequestLoggerStackMiddleware;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Implements a settings form for Reroute Email configuration.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * A TypedConfigManager.
   *
   * @var \Drupal\Core\Config\TypedConfigManagerInterface
   */
  protected TypedConfigManagerInterface $configTyped;

  /**
   * The Request Logger Stack Middleware.
   *
   * @var \Drupal\request_logger\StackMiddleware\RequestLoggerStackMiddleware
   */
  protected RequestLoggerStackMiddleware $requestLoggerStackMiddleware;

  /**
   * The typed Logger settings.
   *
   * @var \Drupal\Core\Config\Schema\Mapping|\Drupal\Core\Config\Schema\Undefined
   */
  private $settingsTyped;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->configTyped = $container->get('config.typed');
    $instance->requestLoggerStackMiddleware = $container->get(RequestLoggerStackMiddleware::class);
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'request_logger_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return [RequestLoggerStackMiddleware::CONFIG_NAME];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $form = parent::buildForm($form, $form_state);
    $config = $this->config(RequestLoggerStackMiddleware::CONFIG_NAME);
    $this->settingsTyped = $this->configTyped->get(RequestLoggerStackMiddleware::CONFIG_NAME);

    $key = RequestLoggerStackMiddleware::CONFIG_KEY_LOG_LEVEL;
    $form[$key] = [
      '#type' => 'select',
      '#title' => $this->getSettingLabel($key),
      '#options' => RfcLogLevel::getLevels(),
      '#default_value' => $config->get($key),
      '#description' => $this->t('Log level of the log entries.'),
      '#config_target' => RequestLoggerStackMiddleware::CONFIG_NAME . ':' . $key,
    ];

    $key = RequestLoggerStackMiddleware::CONFIG_KEY_REQUEST_DATA;
    $form[$key] = [
      '#type' => 'checkboxes',
      '#title' => $this->getSettingLabel($key),
      '#options' => $this->getDataItemOptions($this->requestLoggerStackMiddleware->getRequestDataItems()),
      '#default_value' => $config->get($key),
      '#description' => $this->t('Choose which request data to add to the log entry.'),
      '#config_target' => RequestLoggerStackMiddleware::CONFIG_NAME . ':' . $key,
    ];

    $key = RequestLoggerStackMiddleware::CONFIG_KEY_RESPONSE_DATA;
    $form[$key] = [
      '#type' => 'checkboxes',
      '#title' => $this->getSettingLabel($key),
      '#options' => $this->getDataItemOptions($this->requestLoggerStackMiddleware->getResponseDataItems()),
      '#default_value' => $config->get($key),
      '#description' => $this->t('Choose which request data to add to the log entry.'),
      '#config_target' => RequestLoggerStackMiddleware::CONFIG_NAME . ':' . $key,
    ];

    $key = RequestLoggerStackMiddleware::CONFIG_KEY_MESSAGE_ADD_DATA;
    $form[$key] = [
      '#type' => 'checkbox',
      '#title' => $this->getSettingLabel($key),
      '#default_value' => $config->get($key),
      '#description' => $this->t('Enables a fallback mode that adds the request and respond data directly to the log message string. Required for loggers that do not support structured data. Disable this to optimize performance and decrease the log entry size, if you use structured loggers like OpenTelemetry, Extended Logger, Logger, etc.'),
      '#config_target' => RequestLoggerStackMiddleware::CONFIG_NAME . ':' . $key,
    ];

    $form['message_text'] = [
      '#type' => 'fieldset',
      '#title' => "Message string items",
      '#states' => [
        'visible' => [
          ':input[name="' . $key . '"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $key = RequestLoggerStackMiddleware::CONFIG_KEY_MESSAGE_REQUEST_DATA;
    $form['message_text'][$key] = [
      '#type' => 'checkboxes',
      '#title' => $this->getSettingLabel($key),
      '#options' => $this->getDataItemOptions($this->requestLoggerStackMiddleware->getRequestDataItems()),
      '#default_value' => $config->get($key),
      '#description' => $this->t('Choose which request data to add to the log entry.'),
      '#config_target' => RequestLoggerStackMiddleware::CONFIG_NAME . ':' . $key,
    ];

    $key = RequestLoggerStackMiddleware::CONFIG_KEY_MESSAGE_RESPONSE_DATA;
    $form['message_text'][$key] = [
      '#type' => 'checkboxes',
      '#title' => $this->getSettingLabel($key),
      '#options' => $this->getDataItemOptions($this->requestLoggerStackMiddleware->getResponseDataItems()),
      '#default_value' => $config->get($key),
      '#description' => $this->t('Choose which request data to add to the log entry.'),
      '#config_target' => RequestLoggerStackMiddleware::CONFIG_NAME . ':' . $key,
    ];

    $key = LoggerLogEventSubscriber::CONFIG_KEY_ADD_REQUEST_UUID;
    $form[$key] = [
      '#type' => 'checkbox',
      '#title' => $this->getSettingLabel($key),
      '#default_value' => $config->get($key),
      '#description' => $this->t('Adds the request UUID and the main request UUID to each log entry.'),
      '#config_target' => RequestLoggerStackMiddleware::CONFIG_NAME . ':' . $key,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $getSelectedKeys = function ($values) {
      return array_values(array_filter($values, function ($value) {
        return $value != 0;
      }, ARRAY_FILTER_USE_BOTH));
    };
    $form_state->setValue(
      RequestLoggerStackMiddleware::CONFIG_KEY_REQUEST_DATA,
      $getSelectedKeys($form_state->getValue(RequestLoggerStackMiddleware::CONFIG_KEY_REQUEST_DATA))
    );
    $form_state->setValue(
      RequestLoggerStackMiddleware::CONFIG_KEY_RESPONSE_DATA,
      $getSelectedKeys($form_state->getValue(RequestLoggerStackMiddleware::CONFIG_KEY_RESPONSE_DATA))
    );
    $form_state->setValue(
      RequestLoggerStackMiddleware::CONFIG_KEY_MESSAGE_REQUEST_DATA,
      $getSelectedKeys($form_state->getValue(RequestLoggerStackMiddleware::CONFIG_KEY_MESSAGE_REQUEST_DATA))
    );
    $form_state->setValue(
      RequestLoggerStackMiddleware::CONFIG_KEY_MESSAGE_RESPONSE_DATA,
      $getSelectedKeys($form_state->getValue(RequestLoggerStackMiddleware::CONFIG_KEY_MESSAGE_RESPONSE_DATA))
    );

  }

  /**
   * Gets the data item options.
   *
   * @param array $data
   *   The data items.
   *
   * @return array
   *   The options.
   */
  private function getDataItemOptions($data) {
    $options = [];
    foreach ($data as $key => $item) {
      $options[$key] = $this->t('@label (key: @key)', [
        // The item label is pretty static, therefore can be used in t().
        // phpcs:ignore
        '@label' => $this->t($item['label']),
        '@key' => $key,
      ]);
    }
    return $options;
  }

  /**
   * Gets the label for a setting from typed settings object.
   */
  private function getSettingLabel(string $key, ?string $fallback = NULL): string {
    try {
      $label = $this->settingsTyped->get($key)->getDataDefinition()->getLabel();
    }
    catch (\InvalidArgumentException) {
      $label = $fallback ?: "[$key]";
    }
    return $label;
  }

}
