<?php

namespace Drupal\request_logger\StackMiddleware;

use Drupal\Component\Uuid\UuidInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\page_cache\StackMiddleware\PageCache;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\HttpKernelInterface;

/**
 * Wraps the HttpKernel to log each request and response.
 */
class RequestLoggerStackMiddleware implements HttpKernelInterface {

  /**
   * The request attribute key for the UUID.
   */
  const REQUEST_ATTRIBUTE_UUID = 'request_uuid';
  const REQUEST_ATTRIBUTE_MAIN_UUID = 'request_main_uuid';

  /**
   * Metadata key for request UUID in log entries.
   */
  public const LOG_ENTRY_KEY_REQUEST_UUID = 'request_uuid';

  /**
   * Metadata key for main request UUID in log entries.
   */
  public const LOG_ENTRY_KEY_MAIN_REQUEST_UUID = 'main_request_uuid';

  /**
   * The configuration name.
   */
  const CONFIG_NAME = 'request_logger.settings';
  const CONFIG_KEY_LOG_LEVEL = 'log_level';
  const CONFIG_KEY_REQUEST_DATA = 'request_data';
  const CONFIG_KEY_RESPONSE_DATA = 'response_data';
  const CONFIG_KEY_MESSAGE_ADD_DATA = 'message_add_data';
  const CONFIG_KEY_MESSAGE_REQUEST_DATA = 'message_request_data';
  const CONFIG_KEY_MESSAGE_RESPONSE_DATA = 'message_response_data';

  /**
   * An array of logged requests and responses.
   *
   * @var array
   */
  protected array $requestLoggerRequests = [];

  /**
   * The main request UUID.
   *
   * @var string
   */
  protected string $requestLoggerMainRequestUuid;

  /**
   * Constructs a RequestLoggerStackMiddleware object.
   *
   * @param \Symfony\Component\HttpKernel\HttpKernelInterface $httpKernel
   *   The wrapped http kernel.
   * @param \Drupal\Core\Logger\LoggerChannelInterface $logger
   *   The logger channel.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\Component\Uuid\UuidInterface $uuid
   *   The UUID service.
   */
  public function __construct(
    protected HttpKernelInterface $httpKernel,
    protected LoggerChannelInterface $logger,
    protected ConfigFactoryInterface $configFactory,
    protected UuidInterface $uuid,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function handle(Request $request, $type = self::MAIN_REQUEST, $catch = TRUE): Response {
    $requestUUid = $this->uuid->generate();
    if ($type === self::MAIN_REQUEST) {
      $this->requestLoggerMainRequestUuid = $requestUUid;
    }

    $request->attributes->set(self::REQUEST_ATTRIBUTE_UUID, $requestUUid);
    if (isset($this->requestLoggerMainRequestUuid)) {
      $request->attributes->set(self::REQUEST_ATTRIBUTE_MAIN_UUID, $this->requestLoggerMainRequestUuid);
    }

    $response = $this->httpKernel->handle($request, $type, $catch);

    // @todo Add sampling rate to the settings.
    // @todo Add duration threshold trigger.
    // @todo Add memory usage threshold trigger.
    // @todo Implement logging after the response is done.
    try {
      $this->logRequestAndResponse($request, $response);
    }
    catch (\Exception $e) {
      $this->logger->error('Error logging request and response: @message', ['@message' => $e->getMessage()]);
    }
    $this->requestLoggerRequests[] = [
      'request' => $request,
      'response' => $response,
    ];
    return $response;
  }

  /**
   * Returns the available request data items.
   *
   * @return array
   *   An associative array with the data item key as key and an array with the
   *   label, message_function and value_function as value.
   */
  public function getRequestDataItems() {
    return [
      'uuid' => [
        'label' => 'Request UUID',
        'message_function' => fn ($value) => 'uuid: ' . $value,
        'value_function' => fn ($request) => $request->attributes->get(self::REQUEST_ATTRIBUTE_UUID),
      ],
      'method' => [
        'label' => 'HTTP method',
        'message_function' => fn ($value) => 'method: ' . $value,
        'value_function' => fn ($request) => $request->getMethod(),
      ],
      'path' => [
        'label' => 'Request path',
        'message_function' => fn ($value) => 'path: ' . $value,
        'value_function' => fn ($request) => $request->getPathInfo(),
      ],
      'query' => [
        'label' => 'Query string',
        'message_function' => fn ($value) => 'query: ' . $value,
        'value_function' => fn ($request) => $request->getQueryString(),
      ],
      'headers' => [
        'label' => 'Headers',
        'message_function' => fn ($value) => 'headers: ' . json_encode($value),
        'value_function' => fn ($request) => $request->headers->all(),
      ],
      self::LOG_ENTRY_KEY_MAIN_REQUEST_UUID => [
        'label' => 'Main request UUID',
        'message_function' => fn ($value) => 'main request: ' . $value,
        'value_function' => fn ($request) => $request->attributes->get(self::REQUEST_ATTRIBUTE_MAIN_UUID),
      ],
    ];
  }

  /**
   * Returns the available response data items.
   *
   * @return array
   *   An associative array with the data item key as key and an array with the
   *   label, message_function and value_function as value.
   */
  public function getResponseDataItems() {
    return [
      // @todo Add user id, dynamic cache hit rate.
      'code' => [
        'label' => 'HTTP status code',
        'message_function' => fn ($value) => 'code: ' . $value,
        'value_function' => fn ($response) => $response->getStatusCode(),
      ],
      'size' => [
        'label' => 'Size (KB)',
        'message_function' => fn ($value) => 'size: ' . round($value / 1024, 3) . ' KB',
        'value_function' => fn ($response) => strlen($response->getContent()),
      ],
      // @todo Calculate the normalized duration with CPU usage like in the
      // module rusage_meter.
      'duration' => [
        'label' => 'Request duration (sec)',
        'message_function' => fn ($value) => 'duration: ' . round($value, 3) . ' sec',
        'value_function' => fn ($response) => round(microtime(TRUE) - ($_SERVER['REQUEST_TIME_FLOAT'] ?? microtime(TRUE)), 3),
      ],
      'memory_usage' => [
        'label' => 'Memory usage (MB)',
        'message_function' => fn ($value) => 'memory: ' . round($value / 1024 / 1024, 3) . ' MB',
        'value_function' => fn ($response) => memory_get_usage(),
      ],
      'memory_usage_peak' => [
        'label' => 'Memory usage peak (MB)',
        'message_function' => fn ($value) => 'memory: ' . round($value / 1024 / 1024, 3) . ' MB',
        'value_function' => fn ($response) => memory_get_peak_usage(),
      ],
      'page_cache' => [
        'label' => 'Page cache status (HIT/MISS)',
        'message_function' => fn ($value) => 'page cache: ' . $value,
        'value_function' => fn ($response) => (
          $response->headers->has(PageCache::HEADER)
          ? $response->headers->get(PageCache::HEADER)
          : NULL
        ),
      ],
      'headers' => [
        'label' => 'Headers',
        'message_function' => fn ($value) => 'headers: ' . json_encode($value),
        'value_function' => fn ($response) => $response->headers->all(),
      ],
    ];
  }

  /**
   * Logs the request and response data.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The current request.
   * @param \Symfony\Component\HttpFoundation\Response $response
   *   The current response.
   */
  protected function logRequestAndResponse($request, $response): void {
    $config = $this->configFactory->get(RequestLoggerStackMiddleware::CONFIG_NAME);

    $enabledRequestData = $config->get(RequestLoggerStackMiddleware::CONFIG_KEY_REQUEST_DATA);
    $messageAddDataEnabled = $config->get(RequestLoggerStackMiddleware::CONFIG_KEY_MESSAGE_ADD_DATA);

    $enabledMessageRequestData = $messageAddDataEnabled
      ? $config->get(RequestLoggerStackMiddleware::CONFIG_KEY_MESSAGE_REQUEST_DATA)
      : [];

    $requestDataItems = $this->getRequestDataItems();
    foreach ($requestDataItems as $key => $data) {
      if (in_array($key, $enabledRequestData ?? [])) {
        $requestData[$key] = $data['value_function']($request);
      }
      if (in_array($key, $enabledMessageRequestData)) {
        $value = $requestData[$key] ?? $data['value_function']($request);
        $requestDataLabeled[$key] = $data['message_function']($value);
      }
    }

    $enabledResponseData = $config->get(RequestLoggerStackMiddleware::CONFIG_KEY_RESPONSE_DATA);
    $enabledMessageResponseData = $messageAddDataEnabled
      ? $config->get(RequestLoggerStackMiddleware::CONFIG_KEY_MESSAGE_RESPONSE_DATA) ?? []
      : [];
    $responseDataItems = $this->getResponseDataItems();
    foreach ($responseDataItems as $key => $data) {
      if (in_array($key, $enabledResponseData)) {
        $responseData[$key] = $data['value_function']($response);
      }
      if (in_array($key, $enabledMessageResponseData)) {
        $value = $responseData[$key] ?? $data['value_function']($response);
        $responseDataLabeled[$key] = $data['message_function']($value);
      }
    }

    $message = '{request_label}';
    $context['request_label'] = $request->getMethod() . ' ' . $request->getPathInfo();

    if (isset($requestDataLabeled)) {
      $message .= ' - {request_data}';
      $context['request_data'] = implode(', ', $requestDataLabeled);
    }
    if (isset($responseDataLabeled)) {
      if (isset($context['request_data'])) {
        $message .= ', response: ';
      }
      else {
        $message .= ' - response: ';
      }
      $message .= '{response_data}';
      $context['response_data'] = implode(', ', $responseDataLabeled);
    }

    if (isset($requestData)) {
      $context['metadata']['request'] = $requestData;
    }
    if (isset($responseData)) {
      $context['metadata']['response'] = $responseData;
    }

    $level = $this->configFactory->get(self::CONFIG_NAME)->get('log_level');
    $this->logger->log($level, $message, $context);
  }

}
