<?php

declare(strict_types=1);

namespace Drupal\Tests\request_logger\Kernel;

use Drupal\Component\Uuid\Uuid;
use Drupal\KernelTests\KernelTestBase;
use Drupal\logger\Logger\Logger;
use Drupal\request_logger\StackMiddleware\RequestLoggerStackMiddleware;
use Drupal\test_helpers\TestHelpers;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpKernel\HttpKernelInterface;

/**
 * Test class for the API mocking framework.
 */
class RequestLoggerTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'logger',
    'page_cache',
    'request_logger',
    'request_logger_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installConfig([
      'logger',
      'page_cache',
      'request_logger',
    ]);
  }

  /**
   * {@inheritdoc}
   */
  protected function tearDown(): void {
    restore_error_handler();
    parent::tearDown();
  }

  /**
   * Tests the log entries data.
   */
  public function testLogEntriesData(): void {
    $kernel = $this->container->get('kernel');
    $logFile = $this->loggerSetup();

    $pathCacheable = '/request_logger_test/page_cacheable';
    $pathUncacheable = '/request_logger_test/page_uncacheable';

    $requestCacheable = Request::create($pathCacheable);
    $requestUncacheable = Request::create($pathUncacheable);
    $pageCacheService = $this->container->get('http_middleware.page_cache');
    $kernel->handle($requestCacheable, HttpKernelInterface::MAIN_REQUEST);
    // We have to force clearing the cache id, because it is not reset
    // per each request.
    TestHelpers::setPrivateProperty($pageCacheService, 'cid', NULL);
    $kernel->handle($requestUncacheable, HttpKernelInterface::MAIN_REQUEST);
    TestHelpers::setPrivateProperty($pageCacheService, 'cid', NULL);
    $kernel->handle($requestCacheable, HttpKernelInterface::MAIN_REQUEST);
    TestHelpers::setPrivateProperty($pageCacheService, 'cid', NULL);
    $kernel->handle($requestUncacheable, HttpKernelInterface::MAIN_REQUEST);
    TestHelpers::setPrivateProperty($pageCacheService, 'cid', NULL);
    $kernel->handle($requestCacheable, HttpKernelInterface::MAIN_REQUEST);
    TestHelpers::setPrivateProperty($pageCacheService, 'cid', NULL);
    $kernel->handle($requestUncacheable, HttpKernelInterface::MAIN_REQUEST);

    $logLines = $this->loggerReadEntries($logFile, 'request_logger');
    $cacheableCount = 0;
    foreach ($logLines as $entry) {
      $pageType = match ($entry['metadata']['request']['path']) {
        "/request_logger_test/page_cacheable" => "cacheable",
        "/request_logger_test/page_uncacheable" => "uncacheable",
      };

      if ($pageType == 'cacheable') {
        $cacheableCount++;
      }

      $this->assertTrue(Uuid::isValid($entry['metadata']["request"]["uuid"]));
      $this->assertEquals('GET', $entry['metadata']['request']['method']);
      // We have not enabled headers by default.
      $this->assertArrayNotHasKey('headers', array_keys($entry['metadata']['request']));

      $this->assertEquals(200, $entry['metadata']['response']['code']);
      $this->assertGreaterThan(1, $entry['metadata']['response']['size']);
      $this->assertGreaterThan(0.01, $entry['metadata']['response']['duration']);
      $this->assertGreaterThan(1, $entry['metadata']['response']['memory_usage']);
      $this->assertGreaterThan(1, $entry['metadata']['response']['memory_usage_peak']);
      // We have not enabled headers by default.
      $this->assertArrayNotHasKey('headers', array_keys($entry['metadata']['response']));

      $this->assertEquals(match ($pageType) {
        'cacheable' => $cacheableCount < 3 ? 'MISS' : 'HIT',
        'uncacheable' => 'UNCACHEABLE (response policy)'
      }, $entry['metadata']['response']['page_cache']);

    }
  }

  /**
   * Tests the main request UUID attribute.
   */
  public function testMainRequestUuidAttribute() {
    $kernel = $this->container->get('kernel');
    $logFile = $this->loggerSetup();

    $this->config(RequestLoggerStackMiddleware::CONFIG_NAME)
      ->set(RequestLoggerStackMiddleware::CONFIG_KEY_REQUEST_DATA, [
        'uuid',
        'method',
        'path',
        'query',
        RequestLoggerStackMiddleware::LOG_ENTRY_KEY_MAIN_REQUEST_UUID,
      ])->save();

    $requestMain = Request::create('/path1');
    $kernel->handle($requestMain, HttpKernelInterface::MAIN_REQUEST);

    $requestSub = Request::create('/path2');
    $kernel->handle($requestSub, HttpKernelInterface::SUB_REQUEST);

    $this->container->get('logger.channel.request_logger')->info('Foo');

    $requestMainSecond = Request::create('/path3');
    $kernel->handle($requestMainSecond, HttpKernelInterface::MAIN_REQUEST);

    $logs = $this->loggerReadEntries($logFile, 'request_logger');

    $requestMainFirstUuid = $logs[0]['metadata']["request"]['uuid'];
    $this->assertTrue(Uuid::isValid($logs[0]['metadata']["request"]['uuid']));
    $this->assertTrue(Uuid::isValid($logs[0][RequestLoggerStackMiddleware::LOG_ENTRY_KEY_MAIN_REQUEST_UUID]));
    $this->assertEquals($logs[0]['metadata']["request"]['uuid'], $logs[0][RequestLoggerStackMiddleware::LOG_ENTRY_KEY_MAIN_REQUEST_UUID]);

    $requestSecondUuid = $logs[1]['metadata']["request"]['uuid'];
    $this->assertTrue(Uuid::isValid($logs[1]['metadata']["request"]['uuid']));
    $this->assertTrue(Uuid::isValid($logs[1][RequestLoggerStackMiddleware::LOG_ENTRY_KEY_MAIN_REQUEST_UUID]));
    $this->assertNotEquals($logs[1]['metadata']["request"]['uuid'], $logs[1][RequestLoggerStackMiddleware::LOG_ENTRY_KEY_MAIN_REQUEST_UUID]);

    $this->assertEquals('Foo', $logs[2]['message']);
    $this->assertEquals($requestMainFirstUuid, $logs[2][RequestLoggerStackMiddleware::LOG_ENTRY_KEY_MAIN_REQUEST_UUID]);
    $this->assertEquals($requestSecondUuid, $logs[2][RequestLoggerStackMiddleware::LOG_ENTRY_KEY_REQUEST_UUID]);

    $this->assertTrue(Uuid::isValid($logs[3]['metadata']["request"]['uuid']));
    $this->assertTrue(Uuid::isValid($logs[3][RequestLoggerStackMiddleware::LOG_ENTRY_KEY_MAIN_REQUEST_UUID]));
    $this->assertEquals($logs[3]['metadata']["request"]['uuid'], $logs[3][RequestLoggerStackMiddleware::LOG_ENTRY_KEY_MAIN_REQUEST_UUID]);
    $this->assertNotEquals($logs[3][RequestLoggerStackMiddleware::LOG_ENTRY_KEY_MAIN_REQUEST_UUID], $requestMainFirstUuid);
  }

  /**
   * Sets up a temporary log file for the logger.
   *
   * @return string
   *   The path to the temporary log file.
   */
  private function loggerSetup(): string {
    // @todo Rework this from file to use a memory-based logger.
    $logFile = tempnam(sys_get_temp_dir(), 'request_logger_test_log_file_');
    $this->config(Logger::CONFIG_NAME)
      ->set('targets', [
        0 => [
          'plugin' => 'file',
          'log_level' => 7,
          'configuration' => json_encode([
            'destination' => $logFile,
          ]),
        ],
      ])->save();
    return $logFile;
  }

  /**
   * Reads log entries from a log file.
   *
   * @param string $file
   *   The path to the log file.
   * @param string|null $channel
   *   (optional) The log channel to filter by.
   *
   * @return array
   *   An array of log entries.
   */
  private function loggerReadEntries(string $file, ?string $channel = NULL): array {
    $logLines = file($file, FILE_IGNORE_NEW_LINES);
    foreach ($logLines as $line) {
      $data = json_decode($line, TRUE);
      if ($channel && $data['channel'] !== $channel) {
        continue;
      }
      $logEntries[] = $data;
    }
    unlink($file);
    return $logEntries;
  }

}
