<?php

/**
 * @file
 * Contains \Drupal\require_login\RequireLoginSubscriber.
 */

namespace Drupal\require_login\EventSubscriber;

use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\FilterResponseEvent;
use Symfony\Component\HttpKernel\Event\GetResponseEvent;
use Symfony\Cmf\Component\Routing\RouteObjectInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Subscribe to kernal request event to check authentication.
 */
class RequireLoginSubscriber implements EventSubscriberInterface {

  /**
   * Checks authentication status of visiting user.
   *
   * @param $config object
   *   Object containing module configurations.
   * @param $route_name string
   *   The current pages routing name.
   * @return boolean
   *   Returns TRUE if user is authenticated and FALSE otherwise.
   */
  public function checkUserAuth($config, $route_name) {
    $lang = \Drupal::languageManager()->getCurrentLanguage()->getId();
    $path = \Drupal::service('path.current')->getPath();
    $path = trim(\Drupal::service('path.alias_manager')->getAliasByPath($path, $lang), '/ ');
    $page_settings = \Drupal::config('system.site')->get('page');

    // Compare current URL with customizable excluded paths. Returns TRUE when
    // at least one excluded path matches the current page path.
    $exclude_paths = explode(PHP_EOL, $config->get('excluded_paths'));
    foreach ($exclude_paths as $key => $exclude_path) {
      $exclude_paths[$key] = trim($exclude_path, '/ ');
      if (preg_match('/<front>/i', $exclude_path)) {
        $front_path = \Drupal::service('path.alias_manager')->getAliasByPath($page_settings['front'], $lang);
        $exclude_paths[$key] = trim($front_path, '/ ');
      }
    }
    if (\Drupal::service('path.matcher')->matchPath($path, implode(PHP_EOL, $exclude_paths))) {
      return TRUE;
    }

    // Various checks to determine exceptions for current page. Returns TRUE
    // when at least one check has evaluated as TRUE.
    $checks = array(
      (\Drupal::currentUser()->id() > 0), // Authentication
      ($route_name == 'system.cron'), // Cron
      ($route_name == 'system.timezone'), // Timezone
      ($route_name == 'user.login' || $route_name == 'user.register' || $route_name == 'user.pass'), // User Pages
      (function_exists('drupal_is_cli') && drupal_is_cli()), // Drush
    );
    foreach ($checks as $check) {
      if ($check) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Redirect non-authenticated users to login page.
   */
  public function checkUserRedirect(GetResponseEvent $event) {
    $config = \Drupal::config('require_login.config');
    $route_name = \Drupal::request()->get(RouteObjectInterface::ROUTE_NAME);
    $path = \Drupal::service('path.current')->getPath();
    $configs = \Drupal::config('system.site')->get('page');

    // Exclude specific system paths from redirect to prevent infinite loops
    // when 'view published content' is disabled (not checked).
    $excluded = array(
      '/system/403', $configs['403'],
      '/system/404', $configs['404'],
    );
    if (in_array($path, $excluded)) {
      return;
    }

    // Check user authentication status. Non-authenticated visitors will
    // automatically be redirected to /user/login. Display customizable message
    // to user stating authentication requirements.
    if (!$this->checkUserAuth($config, $route_name)) {
      $message = $config->get('deny_message');
      if (!empty($message)) {
        drupal_set_message(t($config->get('deny_message')));
      }
      $event->setResponse(new RedirectResponse(\Drupal::url('user.login')));
    }
    return;
  }

  /**
   * {@inheritdoc}
   */
  static function getSubscribedEvents() {
    $events[KernelEvents::REQUEST][] = array('checkUserRedirect');
    return $events;
  }
}
