<?php

namespace Drupal\require_login\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configure Require Login settings for this site.
 */
class RequireLoginSettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'require_login_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['require_login.config'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('require_login.config');
    $form['require_login_deny_message'] = array(
      '#type' => 'textarea',
      '#title' => t('Deny Message'),
      '#description' => t('Access denied message displayed after user login page redirect.'),
      '#default_value' => $config->get('deny_message'),
    );
    $form['require_login_excluded_paths'] = array(
      '#type' => 'textarea',
      '#title' => t('Exclude Paths'),
      '#description' => t("Relative paths that shouldn't require user login (one per line). Use &lt;front&gt; for front page."),
      '#default_value' => $config->get('excluded_paths'),
    );
    $form['require_login_auth_path'] = array(
      '#type' => 'textfield',
      '#title' => t('User Login Path'),
      '#description' => t('Relative path to user login page (if different from /user/login). Supports URL queries and fragments.') . '<br />' . t('Example: /login?destination=/node/1'),
      '#default_value' => $config->get('auth_path'),
    );
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $absolutes = array();

    // Confirm exclusion paths are relative. Also trims and confirms presence of
    // leading forward slash on paths. We don't check if page actually exists.
    $exclude_paths = explode(PHP_EOL, $form_state->getValue('require_login_excluded_paths'));
    foreach ($exclude_paths as $key => $exclude_path) {
      $exclude_path = trim($exclude_path);
      $url = parse_url($exclude_path);

      // Detect invalid absolute domain in path.
      if (isset($url['scheme']) || isset($url['host']) || preg_match('/^www./i', $url['path'])) {
        $absolutes[] = trim($exclude_path);
      }

      // Confirm leading forward slash presence.
      elseif (substr($exclude_path, 0, 1) != '/') {
        $exclude_paths[$key] = '/' . $exclude_path;
      }

      // Trim unnecessary whitespace from ends.
      else {
        $exclude_paths[$key] = $exclude_path;
      }
    }
    if ($absolutes) {
      $form_state->setErrorByName('require_login_excluded_paths', $this->t('All exclusion paths must be relative. Invalid paths:<br />!paths', array(
        '!paths' => implode('<br />', $absolutes),
      )));
    }
    $form_state->setValue('require_login_excluded_paths', implode(PHP_EOL, $exclude_paths));

    // Confirm user login path is relative. Also trims and confirms presence
    // of leading forward slash. We don't check if page actually exists.
    $auth_path = trim($form_state->getValue('require_login_auth_path'));
    if (!empty($auth_path)) {
      $url = parse_url($auth_path);

      // Detect invalid absolute domain in path.
      if (isset($url['scheme']) || isset($url['host']) || preg_match('/^www./i', $url['path'])) {
        $form_state->setErrorByName('require_login_auth_path', $this->t('User login path must be relative.'));
      }

      // Confirm leading forward slash presence.
      elseif (substr($auth_path, 0, 1) != '/') {
        $form_state->setValue('require_login_auth_path', '/' . $auth_path);
      }

      // Trim unnecessary whitespace from ends.
      else {
        $form_state->setValue('require_login_auth_path', $auth_path);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $this->config('require_login.config')
      ->set('deny_message', $form_state->getValue('require_login_deny_message'))
      ->set('excluded_paths', $form_state->getValue('require_login_excluded_paths'))
      ->set('auth_path', $form_state->getValue('require_login_auth_path'))
      ->save();

    parent::submitForm($form, $form_state);
  }

}
