<?php

namespace Drupal\Tests\require_on_publish\FunctionalJavascript;

use Drupal\field\Entity\FieldConfig;
use Drupal\taxonomy\Entity\Vocabulary;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Tests the 'Require on Publish' functionality on field config pages.
 *
 * @group require_on_publish
 */
class ConfigPageTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'field_ui',
    'require_on_publish',
    'taxonomy',
  ];

  /**
   * The account to login as.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $account;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    // Create the 'article' content type.
    /** @var \Drupal\node\NodeTypeInterface $type */
    $type = $this->container->get('entity_type.manager')->getStorage('node_type')
      ->create([
        'type' => 'article',
        'name' => 'Article',
      ]);
    $type->save();
    $this->container->get('router.builder')->rebuild();

    Vocabulary::create([
      'vid' => 'tags',
      'description' => 'A test vocabulary.',
      'name' => 'Tags',
    ])->save();

    // Create the 'field_tags' field.
    FieldStorageConfig::create([
      'entity_type' => 'node',
      'field_name' => 'field_tags',
      'type' => 'entity_reference',
      'settings' => [
        'target_type' => 'taxonomy_term',
      ],
    ])->save();

    // Add 'field_tags' to the 'article' content type.
    FieldConfig::create([
      'field_name' => 'field_tags',
      'entity_type' => 'node',
      'bundle' => 'article',
      'settings' => [
        'handler' => 'default:taxonomy_term',
        'handler_settings' => [
          'target_bundles' => [
            'tags' => 'tags',
          ],
        ],
      ],
    ])->save();

    // Create a user who can administer fields.
    $this->account = $this->drupalCreateUser(['administer node fields']);
  }

  /**
   * Logs in and navigates to a field config edit page.
   */
  protected function logIn() {
    $this->drupalLogin($this->account);
    $this->drupalGet('admin/structure/types/manage/article/fields/node.article.field_tags');
  }

  /**
   * Test that the 'Required on Publish' field exists.
   */
  public function testRequireOnPublishExists() {
    $this->logIn();

    // Test that there is a checkbox for 'Required on Publish'.
    $this->assertSession()->fieldExists('require_on_publish');
  }

  /**
   * Test that the 'Required on Publish' field toggle works.
   *
   * 'Required on Publish' and 'Required' should not be able to both be checked
   * at the same time. This test ensures that the Javascript that enforces that
   * is working.
   */
  public function testRequireOnPublishToggles() {
    $this->logIn();

    $this->getSession()->getPage()->fillField('require_on_publish', 1);
    // Verify 'Required' is empty.
    $required = $this->getSession()->getPage()->findField('required')->getValue();
    $this->assertEquals(0, $required);

    // Click 'Required'.
    $this->getSession()->getPage()->fillField('required', 1);
    // Verify 'Required on Publish' is empty.
    $required = $this->getSession()->getPage()->findField('require_on_publish')->getValue();
    $this->assertEquals(0, $required);
  }

  /**
   * Test that the 'Required on Publish' field can be successfully enabled.
   */
  public function testRequireOnPublishEnable() {
    $this->logIn();

    $this->submitForm([
      'require_on_publish' => TRUE,
    ], 'Save settings');

    $this->assertSession()->waitForText('Saved Tags configuration');

    $entity_field_manager = \Drupal::service('entity_field.manager');
    $entity_field_manager->clearCachedFieldDefinitions();
    $field_config = FieldConfig::loadByName('node', 'article', 'field_tags');
    $this->assertTrue($field_config->getThirdPartySetting('require_on_publish', 'require_on_publish', FALSE));
  }

  /**
   * Test that the 'Required on Publish' field can be successfully disabled.
   */
  public function testRequireOnPublishDisable() {
    $this->logIn();

    $this->submitForm([
      'require_on_publish' => FALSE,
    ], 'Save settings');

    $this->assertSession()->waitForText('Saved Tags configuration');

    $entity_field_manager = \Drupal::service('entity_field.manager');
    $entity_field_manager->clearCachedFieldDefinitions();
    $field_config = FieldConfig::loadByName('node', 'article', 'field_tags');
    $this->assertNull($field_config->getThirdPartySetting('require_on_publish', 'require_on_publish'));
  }

  /**
   * Test that the warning toggle works.
   */
  public function testWarningToggle() {
    $this->logIn();
    $this->getSession()->getPage()->checkField('require_on_publish');
    $this->assertSession()->fieldExists('warn_on_empty');
  }

  /**
   * Test that the warning toggle can be enabled.
   */
  public function testWarningToggleEnable() {
    $this->logIn();

    $this->submitForm([
      'require_on_publish' => TRUE,
      'warn_on_empty' => TRUE,
    ], 'Save settings');

    $this->assertSession()->waitForText('Saved Tags configuration');

    $entity_field_manager = \Drupal::service('entity_field.manager');
    $entity_field_manager->clearCachedFieldDefinitions();
    $field_config = FieldConfig::loadByName('node', 'article', 'field_tags');
    $this->assertTrue($field_config->getThirdPartySetting('require_on_publish', 'warn_on_empty', FALSE));
  }

  /**
   * Test that the warning toggle can be disabled.
   */
  public function testWarningToggleDisable() {
    $this->logIn();

    $this->submitForm([
      'require_on_publish' => TRUE,
      'warn_on_empty' => FALSE,
    ], 'Save settings');

    $this->assertSession()->waitForText('Saved Tags configuration');

    $entity_field_manager = \Drupal::service('entity_field.manager');
    $entity_field_manager->clearCachedFieldDefinitions();
    $field_config = FieldConfig::loadByName('node', 'article', 'field_tags');
    $this->assertNull($field_config->getThirdPartySetting('require_on_publish', 'warn_on_empty'));
  }

}
