<?php

namespace Drupal\require_on_publish\Plugin\Validation\Constraint;

use Drupal\node\NodeInterface;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\Core\Field\FieldConfigInterface;
use Symfony\Component\Validator\Constraint;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Symfony\Component\Validator\ConstraintValidator;
use Drupal\content_moderation\ContentModerationState;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\content_moderation\ModerationInformationInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Validates the RequireOnPublish constraint.
 */
class RequireOnPublishValidator extends ConstraintValidator implements ContainerInjectionInterface {

  use StringTranslationTrait;

  /**
   * Whether this validation run should enforce required on publish.
   *
   * @var bool
   */
  protected $enforceRequireOnPublish = FALSE;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    protected ModuleHandlerInterface $moduleHandler,
    protected RequestStack $requestStack,
    protected MessengerInterface $messenger,
    protected ?ModerationInformationInterface $moderationInfo = NULL,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('module_handler'),
      $container->get('request_stack'),
      $container->get('messenger'),
      $container->has('content_moderation.moderation_information')
        ? $container->get('content_moderation.moderation_information')
        : NULL
    );
  }

  /**
   * {@inheritdoc}
   */
  public function validate($entity, Constraint $constraint): void {
    if (!$entity instanceof ContentEntityInterface) {
      return;
    }

    // EARLY EXIT: Skip when the incoming entity being validated is a Paragraph.
    // We only enforce paragraph subfields when we're validating the parent
    // content entity that *contains* the paragraphs.
    if ($entity->getEntityTypeId() === 'paragraph') {
      return;
    }

    // Compute publish intent from the parent content entity being validated.
    $this->enforceRequireOnPublish = $this->isPublishing($entity);

    // Walk all fields on the parent entity.
    foreach ($entity->getFields() as $items) {
      if (!($items instanceof FieldItemListInterface)) {
        continue;
      }
      $field_definition = $items->getFieldDefinition();

      if ($this->isParagraphReferenceField($field_definition)) {
        $this->validateParagraphReferenceField($items, $constraint);
      }
      elseif ($this->isNameField($field_definition)) {
        $this->validateNameField($items, $constraint);
      }
      else {
        $this->validateField($items, $constraint);
      }
    }
  }

  /**
   * Determine publish status, or parent’s status if this is a paragraph.
   */
  protected function isPublishing(ContentEntityInterface $entity): bool {
    // Content Moderation takes precedence when present & applicable.
    if ($this->moderationInfo && $this->moderationInfo->isModeratedEntity($entity) && $entity->hasField('moderation_state')) {
      $state_id = (string) $entity->get('moderation_state')->value;
      if ($state_id !== '') {
        $workflow = $this->moderationInfo->getWorkflowForEntity($entity);
        $state = $workflow?->getTypePlugin()?->getState($state_id);
        if ($state instanceof ContentModerationState) {
          return $state->isPublishedState();
        }
      }
    }

    // Not moderated. Respect 'status' if present; otherwise use isPublished().
    if ($entity->hasField('status')) {
      return (bool) $entity->get('status')->value;
    }
    return $entity instanceof NodeInterface ? $entity->isPublished() : FALSE;
  }

  /**
   * Detects a paragraph reference revisions field.
   */
  protected function isParagraphReferenceField(FieldDefinitionInterface $field_definition): bool {
    return $field_definition->getType() === 'entity_reference_revisions'
      && $field_definition->getSetting('target_type') === 'paragraph';
  }

  /**
   * Enforce/warn on paragraph subfields only when invoked from the parent.
   */
  protected function validateParagraphReferenceField(FieldItemListInterface $items, Constraint $constraint): void {
    $parent_field_name = $items->getFieldDefinition()->getName();

    /** @var \Drupal\paragraphs\Plugin\Field\FieldType\ParagraphItem[] $items */
    foreach ($items as $delta => $item) {
      $paragraph = $item->entity;
      if (!($paragraph instanceof Paragraph)) {
        continue;
      }

      foreach ($paragraph->getFields() as $subfield) {
        $subfield_definition = $subfield->getFieldDefinition();
        if (!($subfield_definition instanceof FieldConfigInterface)) {
          continue;
        }

        if ($this->isNameField($subfield_definition)) {
          // Anchor any violations to the host Paragraphs field item.
          $this->validateNameField($subfield, $constraint, $parent_field_name, $delta);
          continue;
        }

        $label = $subfield_definition->getLabel();
        $sub_name = $subfield_definition->getName();
        $empty = $subfield->isEmpty();
        $req = (bool) $subfield_definition->getThirdPartySetting('require_on_publish', 'require_on_publish', FALSE);
        $warn = (bool) $subfield_definition->getThirdPartySetting('require_on_publish', 'warn_on_empty', FALSE);

        // Not publishing: warnings only (if configured).
        if (!$this->enforceRequireOnPublish) {
          if ($warn && $empty) {
            $this->messenger->addWarning($this->t(
              '@label may be empty until publishing.',
              ['@label' => $label]
            ));
          }
          continue;
        }

        // Published → enforce require_on_publish.
        if ($req && $empty) {
          $this->context
            ->buildViolation($constraint->message, ['@field_label' => $label])
            ->atPath("{$parent_field_name}.{$delta}.subform.{$sub_name}")
            ->addViolation();
        }
      }
    }
  }

  /**
   * Whether the field is a Name field.
   */
  protected function isNameField(FieldDefinitionInterface $definition): bool {
    return $this->moduleHandler->moduleExists('name')
      && $definition->getType() === 'name';
  }

  /**
   * Validate a Name field honoring "require on publish" and "warn on empty".
   *
   * - On publish: enforce that configured minimum components are present.
   * - On draft: optionally warn if empty or missing minimum components.
   *
   * @see name_element_validate()
   */
  protected function validateNameField(FieldItemListInterface $items, Constraint $constraint, ?string $anchorField = NULL, ?int $anchorDelta = NULL): void {
    $field_definition = $items->getFieldDefinition();
    if (!($field_definition instanceof FieldConfigInterface)) {
      return;
    }

    $require_on_publish = (bool) $field_definition->getThirdPartySetting('require_on_publish', 'require_on_publish', FALSE);
    $warn_on_empty = (bool) $field_definition->getThirdPartySetting('require_on_publish', 'warn_on_empty', FALSE);

    // If neither is set, nothing to do.
    if (!$require_on_publish && !$warn_on_empty) {
      return;
    }

    $label = $field_definition->getLabel();
    $settings = $field_definition->getSettings();
    $labels = $settings['labels'] ?? [];
    $minimum_components = array_filter($settings['minimum_components'] ?? []);
    $allow_family_or_given = !empty($settings['allow_family_or_given']);

    $is_empty = $items->isEmpty();
    $present = [];

    if (!$is_empty) {
      $item = $items->first();
      // Use Name’s known component keys via labels array.
      foreach ($labels as $key => $component_label) {
        $value = $item->get($key)->getValue();
        if (!empty($value)) {
          $present[$key] = TRUE;
        }
      }
      // If "allow family OR given" and either is present, treat both as
      // satisfied.
      if ($allow_family_or_given && (isset($present['given']) || isset($present['family']))) {
        $present['given']  = TRUE;
        $present['family'] = TRUE;
      }
    }

    // Compute which minimum components are missing.
    $required_keys = array_keys($minimum_components);
    $missing_keys = $is_empty ? $required_keys : array_values(array_diff($required_keys, array_keys($present)));
    $missing_labels = array_map(fn($k) => $labels[$k] ?? $k, $missing_keys);

    if (!$this->enforceRequireOnPublish) {
      if ($warn_on_empty) {
        if ($is_empty) {
          $this->messenger->addWarning($this->t('@label may be empty until publishing.', ['@label' => $label]));
        }
        elseif ($missing_labels) {
          $this->messenger
            ->addWarning($this->t('The following parts of @label may be empty until publishing: <em>@components</em>.', [
              '@label' => $label,
              '@components' => implode(', ', $missing_labels),
            ]));
        }
      }
      return;
    }

    if ($require_on_publish) {
      // Decide where to attach the violation:
      // - Nested in Paragraphs: anchor to the host Paragraphs field item.
      // - Top-level Name field: use the field machine name.
      $path = ($anchorField !== NULL && $anchorDelta !== NULL)
        ? "{$anchorField}.{$anchorDelta}"
        : $field_definition->getName();

      if ($is_empty) {
        $this->context
          ->buildViolation($constraint->message, ['@field_label' => $label])
          ->atPath($path)
          ->addViolation();
        return;
      }

      if ($missing_labels) {
        $this->context
          ->buildViolation($this->t('@label also requires the following parts when publishing: <em>@components</em>.', [
            '@label' => $label,
            '@components' => implode(', ', $missing_labels),
          ]))
          ->atPath($path)
          ->addViolation();
      }
    }
  }

  /**
   * Validates a scalar/text/boolean/etc. field with ROP settings.
   */
  protected function validateField(FieldItemListInterface $items, Constraint $constraint): void {
    $field_definition = $items->getFieldDefinition();
    if (!($field_definition instanceof FieldConfigInterface)) {
      return;
    }

    $require_on_publish = (bool) $field_definition->getThirdPartySetting('require_on_publish', 'require_on_publish', FALSE);
    $warn_on_empty = (bool) $field_definition->getThirdPartySetting('require_on_publish', 'warn_on_empty', FALSE);

    $states_condition = $items->getEntity()->_requireOnPublish[$items->getName()]['condition'] ?? NULL;
    if (!empty($states_condition)) {
      // Evaluate against the submitted form values (raw input names).
      $require_on_publish = _require_on_publish_evaluate_state_condition(
        $states_condition,
        $this->requestStack->getCurrentRequest()->request->all()
      );
    }

    // If neither is set, nothing to do.
    if (!$require_on_publish && !$warn_on_empty) {
      return;
    }

    $label = $field_definition->getLabel();

    // Empty detection (booleans special).
    $empty = $field_definition->getType() === 'boolean'
      ? !(bool) $items->value
      : $items->isEmpty();

    // Draft/unpublished → only warn.
    if (!$this->enforceRequireOnPublish) {
      if ($warn_on_empty && $empty) {
        $this->messenger->addWarning($this->t(
          '@label may be empty until publishing.',
          ['@label' => $label]
        ));
      }
      return;
    }

    // Published → enforce require_on_publish.
    if ($require_on_publish && $empty) {
      $this->context
        ->buildViolation($constraint->message, ['@field_label' => $label])
        ->atPath($field_definition->getName())
        ->addViolation();
    }
  }

}
