<?php

namespace Drupal\Tests\require_on_publish\FunctionalJavascript;

use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\field\Entity\FieldConfig;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Verify ROP Paragraph subfields on a node form.
 *
 * @group require_on_publish
 */
class ParagraphsTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'user',
    'field',
    'text',
    'entity_reference_revisions',
    'paragraphs',
    'require_on_publish',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    NodeType::create(['type' => 'article', 'name' => 'Article'])->save();
    ParagraphsType::create(['id' => 'note', 'label' => 'Note'])->save();

    // Paragraph subfields.
    FieldStorageConfig::create([
      'field_name' => 'field_subfield_note1',
      'entity_type' => 'paragraph',
      'type' => 'string',
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_subfield_note1',
      'entity_type' => 'paragraph',
      'bundle' => 'note',
      'label' => 'Sub Note 1',
      'third_party_settings' => [
        'require_on_publish' => ['require_on_publish' => TRUE],
      ],
    ])->save();

    FieldStorageConfig::create([
      'field_name' => 'field_subfield_note2',
      'entity_type' => 'paragraph',
      'type' => 'string',
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_subfield_note2',
      'entity_type' => 'paragraph',
      'bundle' => 'note',
      'label' => 'Sub Note 2',
      'third_party_settings' => [
        'require_on_publish' => ['warn_on_empty' => TRUE],
      ],
    ])->save();

    FieldStorageConfig::create([
      'field_name' => 'field_subfield_note3',
      'entity_type' => 'paragraph',
      'type' => 'string',
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_subfield_note3',
      'entity_type' => 'paragraph',
      'bundle' => 'note',
      'label' => 'Sub Note 3',
      'third_party_settings' => [
        'require_on_publish' => ['require_on_publish' => TRUE],
      ],
    ])->save();

    // Node field referencing paragraphs.
    FieldStorageConfig::create([
      'field_name' => 'field_note',
      'entity_type' => 'node',
      'type' => 'entity_reference_revisions',
      'settings' => ['target_type' => 'paragraph'],
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_note',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Note',
      'settings' => [
        'handler' => 'default:paragraph',
        'handler_settings' => [
          'target_bundles' => ['note' => 'note'],
        ],
      ],
    ])->save();

    // Configure form displays.
    /** @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface $repo */
    $repo = \Drupal::service('entity_display.repository');

    // Node form display: use the Paragraphs widget (with revisions).
    $node_display = $repo->getFormDisplay('node', 'article', 'default');
    $node_display->setComponent('field_note', [
      'type' => 'paragraphs',
      'settings' => [
        'edit_mode' => 'open',
        'form_display_mode' => 'default',
        'add_mode' => 'button',
      ],
      'weight' => 10,
    ]);
    $node_display->save();

    // Paragraph form display: ensure textfield widgets are visible.
    $para_display = $repo->getFormDisplay('paragraph', 'note', 'default');
    $para_display->setComponent('field_subfield_note1', ['type' => 'string_textfield', 'weight' => 0]);
    $para_display->setComponent('field_subfield_note2', ['type' => 'string_textfield', 'weight' => 1]);
    $para_display->setComponent('field_subfield_note3', ['type' => 'string_textfield', 'weight' => 1]);
    $para_display->save();

    $this->drupalLogin($this->drupalCreateUser([
      'create article content',
      'edit any article content',
      'administer nodes',
    ]));
  }

  /**
   * Test draft save displays warn on empty.
   */
  public function testDraftSaveWarnsOnly(): void {
    $this->drupalGet('node/add/article');
    $page = $this->getSession()->getPage();

    $page->fillField('Title', 'Article with Empty Note');
    $page->uncheckField('Published');
    $page->pressButton('Save');

    $this->assertSession()->pageTextNotContains('Error message');
    $this->assertSession()->pageTextContains('Sub Note 2 may be empty until publishing.');
  }

  /**
   * Test required on publish.
   */
  public function testPublishEnforcesRequired(): void {
    $this->drupalGet('node/add/article');
    $page = $this->getSession()->getPage();

    $page->fillField('Title', 'Publishing Article');
    $page->checkField('Published');
    $page->pressButton('Save');

    $this->assertSession()->waitForText('Error message');
    $this->assertSession()->pageTextContains('Sub Note 1 field is required when publishing.');
    $this->assertSession()->pageTextContains('Sub Note 3 field is required when publishing.');
  }

  /**
   * Test unpublishing a published entity kicks the error.
   */
  public function testEditToUnpublishSkipsEnforcement(): void {
    // Create a Published node with Sub Note 1 filled so it can be published.
    $paragraph = Paragraph::create([
      'type' => 'note',
      'field_subfield_note1' => 'filled',
      'field_subfield_note3' => 'filled',
    ]);
    $paragraph->save();
    $published_node = Node::create([
      'type'   => 'article',
      'title'  => 'Initially Published',
      'status' => 1,
      'field_note' => [
        [
          'target_id' => $paragraph->id(),
          'target_revision_id' => $paragraph->getRevisionId(),
        ],
      ],
    ]);
    $published_node->save();

    // Edit: clear required subfield and uncheck Published.
    $this->drupalGet('/node/' . $published_node->id() . '/edit');
    $page = $this->getSession()->getPage();
    $page->fillField('Sub Note 1', '');
    $page->uncheckField('Published');
    $page->pressButton('Save');

    // On unpublish, there should be no enforcement error.
    $this->assertSession()->pageTextNotContains('Error message');
    // Warn-on-empty (optional) still allowed.
    $this->assertSession()->pageTextContains('Sub Note 2 may be empty until publishing.');
  }

}
