<?php

namespace Drupal\Tests\require_on_publish\FunctionalJavascript;

use Drupal\node\Entity\NodeType;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Core\Entity\Entity\EntityFormDisplay;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Tests Require on Publish with Form API #states(require_on_publish).
 *
 * @group require_on_publish
 */
class StatesFormApiTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'text',
    'require_on_publish',
    'require_on_publish_test_states',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Content type.
    NodeType::create(['type' => 'article', 'name' => 'Article'])->save();

    // Gate: boolean checkbox.
    FieldStorageConfig::create([
      'field_name' => 'field_gate',
      'entity_type' => 'node',
      'type' => 'boolean',
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_gate',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Gate',
      'settings' => [
        'on_label' => 'On',
        'off_label' => 'Off',
      ],
    ])->save();

    // Target: simple string field that will be ROP-required via #states.
    FieldStorageConfig::create([
      'field_name' => 'field_target',
      'entity_type' => 'node',
      'type' => 'string',
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_target',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Target',
    ])->save();

    $form_display = EntityFormDisplay::load('node.article.default');
    if (!$form_display) {
      $form_display = EntityFormDisplay::create([
        'targetEntityType' => 'node',
        'bundle' => 'article',
        'mode' => 'default',
        'status' => TRUE,
        'id' => 'node.article.default',
      ]);
    }

    // Put the fields on the form with sensible widgets.
    $form_display->setComponent('field_gate', [
      'type' => 'boolean_checkbox',
      'weight' => 20,
    ]);
    $form_display->setComponent('field_target', [
      'type' => 'string_textfield',
      'weight' => 21,
    ]);

    $form_display->save();

    // Author with publish permission.
    $account = $this->createUser([
      'access content',
      'create article content',
      'edit own article content',
      'view own unpublished content',
      'administer nodes',
    ]);
    $this->drupalLogin($account);
  }

  /**
   * JS: indicator toggles when the #states condition flips.
   */
  public function testIndicatorToggles(): void {
    $this->drupalGet('node/add/article');

    // Initially, the target label should NOT carry the ROP indicator class.
    $this->assertSession()
      ->elementExists('css', 'label[for^="edit-field-target-0-value"]');
    $this->assertSession()
      ->elementNotExists('css', 'label.form-required-on-publish[for^="edit-field-target-0-value"]');

    // Flip the gate checkbox to TRUE.
    $page = $this->getSession()->getPage();
    $page->checkField('field_gate[value]');

    // Wait for states to fire and class to be added.
    $this->assertSession()
      ->waitForElement('css', 'label.form-required-on-publish[for^="edit-field-target-0-value"]');
    $this->assertSession()
      ->elementExists('css', 'label.form-required-on-publish[for^="edit-field-target-0-value"]');
  }

  /**
   * Server-side: publish passes when state is OFF, fails when state is ON.
   */
  public function testServerValidationHonorsStates(): void {
    // Can publish without Target.
    $this->drupalGet('node/add/article');
    $this->submitForm([
      'title[0][value]' => 'States OFF',
      'status[value]' => 1,
    ], 'Save');

    // Should NOT see the ROP error for Target.
    $this->assertSession()
      ->pageTextNotContains('Target field is required when publishing.');

    // Publishing without Target should fail with ROP error.
    $this->drupalGet('node/add/article');

    // Intentionally omit field_target.
    $this->submitForm([
      'title[0][value]' => 'States ON',
      'status[value]' => 1,
      'field_gate[value]' => 1,
    ], 'Save');

    $this->assertSession()
      ->pageTextContains('Target field is required when publishing.');

    // Gate checked AND Target filled => publish succeeds.
    $this->submitForm([
      'title[0][value]' => 'States ON filled',
      'field_target[0][value]' => 'Filled',
      'status[value]' => 1,
    ], 'Save');

    $this->assertSession()
      ->pageTextNotContains('Target field is required when publishing.');
    $this->assertSession()
      ->pageTextContains('has been created');

  }

}
