<?php

namespace Drupal\Tests\require_on_publish\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\Entity\FieldConfig;

/**
 * Ensures ROP enforces Name minimum components inside Paragraphs.
 *
 * @group require_on_publish
 */
class ParagraphsNameComponentsValidatorTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'field',
    'filter',
    'text',
    'file',
    'node',
    'entity_reference_revisions',
    'paragraphs',
    'name',
    'require_on_publish',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installSchema('node', 'node_access');
    $this->installEntitySchema('paragraph');
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');

    // Content type.
    NodeType::create(['type' => 'article', 'name' => 'Article'])->save();

    // Paragraph type.
    ParagraphsType::create(['id' => 'person_note', 'label' => 'Person Note'])->save();

    // Name field storage on Paragraphs.
    FieldStorageConfig::create([
      'field_name' => 'field_person_name',
      'entity_type' => 'paragraph',
      'type' => 'name',
      'cardinality' => 1,
    ])->save();

    // Name field instance on 'person_note' with Name's own minimum_components.
    $name_field = FieldConfig::create([
      'field_name' => 'field_person_name',
      'entity_type' => 'paragraph',
      'bundle' => 'person_note',
      'label' => 'Person name',
      // 'settings' => [
      // // // Name module's built-in component requirement.
      // // 'minimum_components' => ['given', 'family'],
      // ],
    ]);
    // Enable ROP enforcement on publish for this field.
    $name_field->setThirdPartySetting('require_on_publish', 'require_on_publish', TRUE);
    // @todo Add warn_on_empty coverage.
    // $name_field->setThirdPartySetting(
    // //  'require_on_publish',
    // //  'warn_on_empty',
    // //  TRUE
    // );
    $name_field->save();

    // Paragraphs reference field on Node.
    FieldStorageConfig::create([
      'field_name' => 'field_notes',
      'entity_type' => 'node',
      'type' => 'entity_reference_revisions',
      'settings' => ['target_type' => 'paragraph'],
      'cardinality' => -1,
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_notes',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Notes',
      'settings' => [
        'handler' => 'default:paragraph',
        'handler_settings' => [
          'target_bundles' => ['person_note' => 'person_note'],
        ],
      ],
    ])->save();
  }

  /**
   * Helper: node with one Paragraph item and a Name value.
   *
   * @param array $name_value
   *   Components keyed by 'given','family', etc.
   * @param bool $published
   *   Node status (TRUE => publish).
   */
  protected function createNodeWithNameInParagraph(array $name_value, bool $published): Node {
    $paragraph = Paragraph::create([
      'type' => 'person_note',
      'field_person_name' => [$name_value],
    ]);
    $paragraph->save();

    return Node::create([
      'type' => 'article',
      'title' => 'Has person note',
      'status' => $published ? 1 : 0,
      'field_notes' => [
        [
          'target_id' => $paragraph->id(),
          'target_revision_id' => $paragraph->getRevisionId(),
        ],
      ],
    ]);
  }

  /**
   * Draft save should NOT enforce ROP minimum components.
   */
  public function testDraftDoesNotEnforceComponents(): void {
    $node = $this->createNodeWithNameInParagraph(['given' => 'Ada'], FALSE);
    $violations = $node->validate();
    $this->assertCount(
      0,
      $violations,
      'Draft node does not enforce Name minimum components.'
    );
  }

  /**
   * Publish with empty Name will show violation listing missing components.
   */
  public function testPublishEmptyNameProducesViolation(): void {
    $node = $this->createNodeWithNameInParagraph([], TRUE);
    $violations = $node->validate();
    $this->assertGreaterThanOrEqual(
      1,
      count($violations),
      'Publishing with empty Name in Paragraph yields violation.'
    );

    $messages = array_map(fn($v) => $v->getMessage()->render(), iterator_to_array($violations));
    $this->assertStringContainsString('Person name field is required when publishing.', $messages[0]);
  }

  /**
   * Publish with only one filled component should show violation.
   */
  public function testPublishPartialNameStillViolates(): void {
    $node = $this->createNodeWithNameInParagraph(['given' => 'Ada'], TRUE);
    $violations = $node->validate();
    $this->assertGreaterThanOrEqual(
      1,
      count($violations),
      'Publishing with partial Name in Paragraph yields violation.'
    );

    $messages = array_map(fn($v) => $v->getMessage()->render(), iterator_to_array($violations));
    $this->assertStringContainsString('Person name also requires the following parts when publishing:', $messages[0]);
  }

  /**
   * Publish with all required components passes validation.
   */
  public function testPublishCompleteNamePasses(): void {
    $node = $this->createNodeWithNameInParagraph(['given' => 'Ada', 'family' => 'Lovelace'], TRUE);
    $violations = $node->validate();
    $this->assertCount(0, $violations, 'Publishing with complete Name components passes validation.');
  }

}
