<?php

namespace Drupal\Tests\require_on_publish\Kernel;

use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\field\Entity\FieldConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\Tests\content_moderation\Traits\ContentModerationTestTrait;

/**
 * Test Content Moderation.
 *
 * @group require_on_publish
 */
class RequireOnPublishContentModerationValidatorTest extends KernelTestBase {
  use ContentModerationTestTrait;
  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'user',
    'field',
    'text',
    'filter',
    'system',
    'workflows',
    'content_moderation',
    'require_on_publish',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install schemas / configs needed.
    $this->installSchema('node', 'node_access');
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installEntitySchema('content_moderation_state');
    $this->installConfig(['content_moderation']);

    // Create the article type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    // Create and attach the editorial workflow to node.article.
    $workflow = $this->createEditorialWorkflow();
    $this->addEntityTypeAndBundleToWorkflow($workflow, 'node', 'article');
    $workflow->save();

    // Add a require-on-publish field.
    FieldStorageConfig::create([
      'field_name' => 'field_rop_text',
      'entity_type' => 'node',
      'type' => 'text',
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_rop_text',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Required on publish text',
      'third_party_settings' => [
        'require_on_publish' => [
          'require_on_publish' => TRUE,
        ],
      ],
    ])->save();
  }

  /**
   * Test draft violation.
   */
  public function testDraftDoesNotTriggerViolation() {
    $node = Node::create([
      'type' => 'article',
      'title' => 'Moderated draft',
    ]);

    $this->setCurrentUser($this->createUser(['use editorial transition create_new_draft']));
    $node->moderation_state->value = 'draft';
    $node->save();
    $violations = $node->validate();
    $this->assertCount(0, $violations);
  }

  /**
   * Test published violation.
   */
  public function testLiveStateTriggersViolation() {
    // The core editorial workflow uses 'published' as the published state; if
    // you renamed it you'd adapt accordingly.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Moderated publish',
    ]);
    $this->setCurrentUser($this->createUser(['use editorial transition publish']));
    $node->moderation_state->value = 'published';
    $node->save();
    $violations = $node->validate();
    $this->assertGreaterThan(0, $violations->count());

    $messages = array_map(fn($v) => $v->getMessage()->render(), iterator_to_array($violations));
    $this->assertStringContainsString('Required on publish text field is required when publishing.', reset($messages));
  }

  /**
   * Test published passes validation.
   */
  public function testPublishedWithFieldPasses() {
    $node = Node::create([
      'type' => 'article',
      'title' => 'With required field',
      'field_rop_text' => 'Filled in',
    ]);
    $this->setCurrentUser($this->createUser(['use editorial transition publish']));
    $node->moderation_state->value = 'published';
    $node->save();
    $violations = $node->validate();
    $this->assertCount(0, $violations);
  }

}
