<?php

namespace Drupal\Tests\require_on_publish\Kernel;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;

/**
 * Tests the RequireOnPublishValidator without moderation.
 *
 * @group require_on_publish
 */
class RequireOnPublishValidatorTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'user',
    'field',
    'text',
    'filter',
    'system',
    'require_on_publish',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('node');
    $this->installEntitySchema('user');
    $this->installSchema('system', ['sequences']);

    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    // Create a field.
    FieldStorageConfig::create([
      'field_name' => 'field_rop_text',
      'entity_type' => 'node',
      'type' => 'text',
      'cardinality' => 1,
      'settings' => [],
    ])->save();

    FieldConfig::create([
      'field_name' => 'field_rop_text',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Required on publish text',
      'required' => FALSE,
      'third_party_settings' => [
        'require_on_publish' => [
          'require_on_publish' => TRUE,
        ],
      ],
    ])->save();
  }

  /**
   * Test unpublished validation.
   */
  public function testValidWhenUnpublished() {
    $node = Node::create([
      'type' => 'article',
      'title' => 'Example node',
      'status' => 0,
    ]);
    $violations = $node->validate();
    $this->assertCount(0, $violations, 'Unpublished node does not require field_rop_text.');
  }

  /**
   * Test published and missing validation.
   */
  public function testViolationWhenPublishedAndMissingField() {
    $node = Node::create([
      'type' => 'article',
      'title' => 'Example node',
      'status' => 1,
    ]);
    $violations = $node->validate();

    $messages = array_map(fn($v) => $v->getMessage()->render(), iterator_to_array($violations));
    $this->assertStringContainsString('Required on publish text field is required when publishing.', reset($messages));
  }

  /**
   * Test published and filled validation.
   */
  public function testValidWhenPublishedAndFieldFilled() {
    $node = Node::create([
      'type' => 'article',
      'title' => 'Example node',
      'status' => 1,
      'field_rop_text' => 'This field is required when publishing',
    ]);
    $violations = $node->validate();
    $this->assertCount(0, $violations, 'Published node with field_rop_text passes validation.');
  }

}
