<?php

namespace Drupal\Tests\require_on_publish\FunctionalJavascript;

use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\FunctionalJavascriptTests\WebDriverTestBase;

/**
 * Tests Require on Publish with the Name field.
 *
 * @group require_on_publish
 */
class NameFieldTest extends WebDriverTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'name',
    'node',
    'require_on_publish',
    'taxonomy',
    'text',
  ];

  /**
   * The account to login as.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $account;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $bundle = 'article';

    /** @var \Drupal\node\NodeTypeInterface $type */
    $type = $this->container->get('entity_type.manager')->getStorage('node_type')
      ->create([
        'type' => $bundle,
        'name' => 'Article',
      ]);
    $type->save();

    /** @var \Drupal\Core\Entity\Display\EntityFormDisplayInterface $entity_form */
    $entity_form = $this->container->get('entity_type.manager')
      ->getStorage('entity_form_display')
      ->create([
        'targetEntityType' => 'node',
        'bundle' => $bundle,
        'mode' => 'default',
        'status' => TRUE,
      ]);

    $field_type_id = 'name_default';
    $field_type = 'name';
    $field_name = $field_type_id . '__' . $field_type;

    FieldStorageConfig::create([
      'entity_type' => 'node',
      'field_name' => $field_name,
      'type' => $field_type,
    ])->save();

    FieldConfig::create([
      'field_name' => $field_name,
      'label' => $field_name,
      'entity_type' => 'node',
      'bundle' => $bundle,
      'third_party_settings' => [
        'require_on_publish' => [
          'require_on_publish' => TRUE,
        ],
      ],
    ])->save();
    $entity_form->setComponent($field_name, ['type' => $field_type_id])->save();

    drupal_flush_all_caches();
    // Permission "administer nodes" grants Publish capability.
    $this->account = $this->drupalCreateUser([
      'create article content',
      'edit any article content',
      'edit own article content',
      'view own unpublished content',
      'administer nodes',
    ]);
  }

  /**
   * Ensures the requires parts when publishing message.
   */
  public function testNameFieldValidationOnPublish() {
    $this->drupalLogin($this->account);
    $this->drupalGet('node/add/article');

    $this->submitForm([
      'title[0][value]' => 'Test Article',
      'status[value]' => 1,
    ], 'Save');

    $this->assertSession()->waitForText('Error message');

    $field_label = 'name_default__name';

    $this->assertSession()->pageTextContains($field_label . ' field is required when publishing');

    $this->submitForm([
      'title[0][value]' => 'Test Article',
      'name_default__name[0][given]' => 'Published Partial',
      'status[value]' => 1,
    ], 'Save');

    $this->assertSession()->waitForText('also requires the following parts');

    $this->assertSession()->pageTextContains('name_default__name also requires the following parts when publishing: Family.');
  }

  /**
   * Warns when Name is empty and saving as unpublished.
   */
  public function testNameFieldWarnOnEmptyWithMinimumComponents(): void {
    // Enable warn_on_empty and require both Given + Family as minimum.
    $field = FieldConfig::loadByName('node', 'article', 'name_default__name');
    $settings = $field->getSettings();
    // $settings['minimum_components'] = ['given' => 1, 'family' => 1];
    // $settings['allow_family_or_given'] = FALSE;
    $field->set('settings', $settings);
    $field->setThirdPartySetting('require_on_publish', 'warn_on_empty', TRUE);
    $field->save();

    $this->drupalLogin($this->account);

    // Draft + completely empty → warning.
    $this->drupalGet('node/add/article');
    $this->submitForm([
      'title[0][value]' => 'Warn Draft - Empty',
      'status[value]' => 0,
    ], 'Save');
    $this->assertSession()->waitForText('Warning message');
    $this->assertSession()->pageTextContains(
      'name_default__name may be empty until publishing.'
    );

    // Draft + partially filled (given only) → no warning (not empty).
    $this->drupalGet('node/add/article');
    $this->submitForm([
      'title[0][value]' => 'Warn Draft - Partial',
      'name_default__name[0][given]' => 'OnlyGiven',
      'status[value]' => 0,
    ], 'Save');

    // Publish + partially filled → violation lists the missing part(s).
    $this->assertSession()->waitForText('Warning message');
    $content = $this->getSession()->getPage()->getContent();
    $test = 'test';
    $this->assertSession()->pageTextContains(
      'The following parts of name_default__name may be empty until publishing: Family.'
    );
  }

}
