<?php

namespace Drupal\Tests\require_on_publish\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\paragraphs\Entity\ParagraphsType;
use Drupal\paragraphs\Entity\Paragraph;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\field\Entity\FieldConfig;

/**
 * Test Paragraphs support.
 *
 * @group require_on_publish
 */
class ParagraphsValidatorTest extends KernelTestBase {

  use UserCreationTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'user',
    'system',
    'filter',
    'text',
    'field',
    'file',
    'entity_reference_revisions',
    'paragraphs',
    'require_on_publish',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Install required schemas and config.
    $this->installSchema('node', 'node_access');
    $this->installSchema('file', 'file_usage');
    $this->installEntitySchema('paragraph');
    $this->installEntitySchema('node');
    $this->installEntitySchema('user');

    // Create article content type.
    NodeType::create(['type' => 'article', 'name' => 'Article'])->save();

    // Create paragraph type “note”.
    ParagraphsType::create(['id' => 'note', 'label' => 'Note'])->save();

    // Sub-field #1: require_on_publish.
    FieldStorageConfig::create([
      'field_name' => 'field_subfield_note1',
      'entity_type' => 'paragraph',
      'type' => 'string',
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_subfield_note1',
      'entity_type' => 'paragraph',
      'bundle' => 'note',
      'label' => 'Sub Note 1',
      'required' => FALSE,
      'third_party_settings' => [
        'require_on_publish' => ['require_on_publish' => TRUE],
      ],
    ])->save();

    // Sub-field #2: warn_on_empty.
    FieldStorageConfig::create([
      'field_name' => 'field_subfield_note2',
      'entity_type' => 'paragraph',
      'type' => 'string',
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_subfield_note2',
      'entity_type' => 'paragraph',
      'bundle' => 'note',
      'label' => 'Sub Note 2',
      'required' => FALSE,
      'third_party_settings' => [
        'require_on_publish' => ['warn_on_empty' => TRUE],
      ],
    ])->save();

    // Sub-field #3: require_on_publish.
    FieldStorageConfig::create([
      'field_name' => 'field_subfield_note3',
      'entity_type' => 'paragraph',
      'type' => 'string',
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_subfield_note3',
      'entity_type' => 'paragraph',
      'bundle' => 'note',
      'label' => 'Sub Note 3',
      'required' => FALSE,
      'third_party_settings' => [
        'require_on_publish' => ['require_on_publish' => TRUE],
      ],
    ])->save();

    // Paragraph reference revisions field on node.article.
    FieldStorageConfig::create([
      'field_name' => 'field_note',
      'entity_type' => 'node',
      'type' => 'entity_reference_revisions',
      'settings' => ['target_type' => 'paragraph'],
      'cardinality' => 1,
    ])->save();
    FieldConfig::create([
      'field_name' => 'field_note',
      'entity_type' => 'node',
      'bundle' => 'article',
      'label' => 'Note',
      'settings' => [
        'target_type' => 'paragraph',
        'handler' => 'default:paragraph',
      ],
    ])->save();
  }

  /**
   * Helper to create a node with one paragraph.
   *
   * @param string|null $val1
   *   Value for field_subfield_note1, or NULL.
   * @param string|null $val2
   *   Value for field_subfield_note2, or NULL.
   * @param string|null $val3
   *   Value for field_subfield_note3, or NULL.
   * @param bool $status
   *   Value for published or unpublished (default).
   *
   * @return \Drupal\node\Entity\Node
   *   The created node.
   */
  private function createNodeWithParagraph(?string $val1 = NULL, ?string $val2 = NULL, ?string $val3 = NULL, bool $status = FALSE): Node {
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test note',
      'status' => $status,
    ]);

    $paragraph = Paragraph::create([
      'type' => 'note',
      'field_subfield_note1' => $val1,
      'field_subfield_note2' => $val2,
      'field_subfield_note3' => $val3,
    ]);
    $paragraph->save();

    // Reference it with both target_id and target_revision_id.
    $node->set('field_note', [
      [
        'target_id' => $paragraph->id(),
        'target_revision_id' => $paragraph->getRevisionId(),
      ],
    ]);

    $node->save();

    return $node;
  }

  /**
   * Draft parent → only warn_on_empty should fire (no violations).
   */
  public function testDraftParagraphWarning(): void {
    $node = $this->createNodeWithParagraph(NULL, NULL, NULL, FALSE);
    $violations = $node->validate();
    $this->assertCount(0, $violations, 'Draft parent causes no ROP violations.');

    // But a warning was emitted for Sub Note 2.
    $warnings = $this->container->get('messenger')->messagesByType('warning');
    $this->assertStringContainsString('Sub Note 2 may be empty until publishing.', reset($warnings));
  }

  /**
   * Published parent → require_on_publish should produce a violation.
   */
  public function testPublishedParentTriggersParagraphViolation(): void {
    $node = $this->createNodeWithParagraph(NULL, NULL, NULL, TRUE);
    $violations = $node->validate();

    $this->assertCount(2, $violations, 'Published parent with missing sub-fields produces two violations.');

    $messages = array_map(fn($v) => $v->getMessage()->render(), iterator_to_array($violations));
    $this->assertStringContainsString('Sub Note 1 field is required when publishing.', $messages[0]);
    $this->assertStringContainsString('Sub Note 3 field is required when publishing.', $messages[1]);
  }

  /**
   * Published parent with both sub-fields filled → no violations.
   */
  public function testPublishedParentWithParagraphFilledPasses(): void {
    $node = $this->createNodeWithParagraph('foo', 'bar', 'baz', TRUE);
    $violations = $node->validate();
    $this->assertCount(0, $violations, 'Published parent with all sub-fields filled passes validation.');
  }

  /**
   * Unpublish a content entity skips enforcement.
   */
  public function testUnpublishInSameSubmitSkipsEnforcement(): void {
    // Start published and valid.
    $node = $this->createNodeWithParagraph('filled', 'filled', 'filled', TRUE);

    // Clear required subfield and move to draft in this validation run.
    /** @var \Drupal\paragraphs\Entity\Paragraph $p */
    $paragraph = $node->get('field_note')->entity;
    $paragraph->set('field_subfield_note1', NULL);
    $paragraph->set('field_subfield_note3', NULL);
    $paragraph->save();

    $node->set('status', 0);
    $violations = $node->validate();

    $this->assertCount(0, $violations, 'Unpublishing in the same submit does not enforce ROP.');
  }

  /**
   * Does not enforce standalone Paragraphs.
   */
  public function testStandaloneParagraphDoesNotEnforce(): void {
    $paragraph = Paragraph::create([
      'type' => 'note',
      // Leave required-on-publish subfield empty.
    ]);
    $paragraph->save();

    $violations = $paragraph->validate();
    $this->assertCount(0, $violations, 'Standalone Paragraph validation does not enforce ROP.');
  }

}
