/**
 * @file
 * Admin behavior for Require Revision module.
 */

(function ($, Drupal, drupalSettings, once) {
  'use strict';

  Drupal.behaviors.requireRevision = {
    attach: function (context, settings) {
      if (!drupalSettings.requireRevision || !drupalSettings.requireRevision.entityTypes) {
        return;
      }

      var entityTypes = drupalSettings.requireRevision.entityTypes;

      // Process each entity type
      $.each(entityTypes, function(entityType, types) {
        if (types.length === 0) {
          return;
        }

        var selectAllSelector = 'input[name="select_all_' + entityType + '"]';
        var isSelectAllChanging = false; // Flag to track if "Select all" is changing

        // Handle "Select all" button
        once('sa-' + entityType, selectAllSelector, context).forEach(function(element) {
          $(element).on('change', function() {
            var checked = $(this).prop('checked');
            isSelectAllChanging = true; // Set flag to prevent updateSelectAllState from running
            
            // Check/uncheck all revision checkboxes and trigger change to update #states
            $.each(types, function(index, typeId) {
              $('input[name="revision_required_' + entityType + '[' + typeId + ']"]', context)
                .prop('checked', checked)
                .trigger('change');
            });
            
            // Wait a moment for #states to process, then check/uncheck revision log checkboxes
            setTimeout(function() {
              $.each(types, function(index, typeId) {
                var $logCheckbox = $('input[name="revision_log_required_' + entityType + '[' + typeId + ']"]', context);
                if (!$logCheckbox.prop('disabled')) {
                  $logCheckbox.prop('checked', checked);
                }
              });
              isSelectAllChanging = false; // Clear flag after everything is done
            }, 100);
          });
        });

        // Handle individual revision checkboxes - update "Select all" state
        $.each(types, function(index, typeId) {
          var revisionCheckboxSelector = 'input[name="revision_required_' + entityType + '[' + typeId + ']"]';
          once('rc-' + entityType + '-' + typeId, revisionCheckboxSelector, context).forEach(function(element) {
            $(element).on('change', function() {
              if (!isSelectAllChanging) {
                updateSelectAllState(entityType, types, context);
              }
            });
          });
        });

        // Handle individual revision log checkboxes - update "Select all" state
        $.each(types, function(index, typeId) {
          var logCheckboxSelector = 'input[name="revision_log_required_' + entityType + '[' + typeId + ']"]';
          once('rlc-' + entityType + '-' + typeId, logCheckboxSelector, context).forEach(function(element) {
            $(element).on('change', function() {
              if (!isSelectAllChanging) {
                updateSelectAllState(entityType, types, context);
              }
            });
          });
        });
      });

      /**
       * Update the "Select all" checkbox state based on individual checkboxes.
       */
      function updateSelectAllState(entityType, types, context) {
        var allChecked = true;
        
        // Check all revision checkboxes
        $.each(types, function(i, typeId) {
          if (!$('input[name="revision_required_' + entityType + '[' + typeId + ']"]', context).prop('checked')) {
            allChecked = false;
            return false; // Break the loop
          }
        });
        
        // If all revisions are checked, also check all revision log checkboxes
        if (allChecked) {
          $.each(types, function(i, typeId) {
            var $logCheckbox = $('input[name="revision_log_required_' + entityType + '[' + typeId + ']"]', context);
            // Only check enabled checkboxes
            if (!$logCheckbox.prop('disabled') && !$logCheckbox.prop('checked')) {
              allChecked = false;
              return false; // Break the loop
            }
          });
        }
        
        // Update the "Select all" checkbox state
        $('input[name="select_all_' + entityType + '"]', context).prop('checked', allChecked);
      }
    }
  };

})(jQuery, Drupal, drupalSettings, once);