<?php

/**
 * @file
 * Contains tests for the Require Revision settings form.
 *
 * This file provides functional tests for the settings form.
 */

namespace Drupal\Tests\require_revision\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests the Require Revision settings form.
 *
 * @group require_revision
 */
class RequireRevisionSettingsFormTest extends BrowserTestBase
{

    /**
     * {@inheritdoc}
     */
    protected $defaultTheme = 'stark';

    /**
     * {@inheritdoc}
     */
    protected static $modules = [
    'require_revision',
    'node',
    'block_content',
    'taxonomy',
    ];

    /**
     * A user with permission to administer the module.
     *
     * @var \Drupal\user\UserInterface
     */
    protected $adminUser;

    /**
     * {@inheritdoc}
     */
    protected function setUp(): void
    {
        parent::setUp();

        // Create an admin user with the necessary permissions.
        $this->adminUser = $this->drupalCreateUser(
            [
            'administer require revision',
            'administer content types',
            'administer blocks',
            'administer taxonomy',
            ]
        );
    }

    /**
     * Tests that the settings form is accessible.
     */
    public function testSettingsFormAccess()
    {
        // Anonymous users should not have access.
        $this->drupalGet('admin/config/content/require_revision');
        $this->assertSession()->statusCodeEquals(403);

        // Admin users should have access.
        $this->drupalLogin($this->adminUser);
        $this->drupalGet('admin/config/content/require_revision');
        $this->assertSession()->statusCodeEquals(200);
        $this->assertSession()
            ->pageTextContains('Select which entity types should require revisions');
    }

    /**
     * Tests saving revision requirements for content types.
     */
    public function testSaveContentTypeSettings()
    {
        // Create a content type.
        $this->drupalCreateContentType(
            [
            'type' => 'article',
            'name' => 'Article',
            ]
        );

        $this->drupalLogin($this->adminUser);
        $this->drupalGet('admin/config/content/require_revision');

        // Open the content types section.
        $this->assertSession()->buttonExists('Content Types');

        // Check that the article checkbox exists.
        $this->assertSession()->fieldExists('revision_required_node[article]');

        // Enable revision requirement for articles.
        $edit = [
        'revision_required_node[article]' => true,
        ];
        $this->submitForm($edit, 'Save configuration');

        // Verify the configuration was saved.
        $config = $this->config('require_revision.settings');
        $revision_required = $config->get('revision_required_node');
        $this->assertTrue(isset($revision_required['article']));
        $this->assertEquals('article', $revision_required['article']);
    }

    /**
     * Tests that revision is enforced on node forms.
     */
    public function testRevisionEnforcementOnNodeForm()
    {
        // Create a content type and configure it to require revisions.
        $this->drupalCreateContentType(
            [
            'type' => 'page',
            'name' => 'Page',
            ]
        );

        $config = $this->config('require_revision.settings');
        $config->set('revision_required_node', ['page' => 'page']);
        $config->save();

        // Create a user who can create pages.
        $user = $this->drupalCreateUser(
            [
            'create page content',
            'edit own page content',
            ]
        );
        $this->drupalLogin($user);

        // Create a node.
        $node = $this->drupalCreateNode(
            [
            'type' => 'page',
            'title' => 'Test Page',
            ]
        );

        // Visit the edit form.
        $this->drupalGet('node/' . $node->id() . '/edit');

        // Check that the revision checkbox is present and disabled.
        $revision_field = $this->getSession()->getPage()->findField('revision');
        $this->assertNotNull($revision_field);
        $this->assertTrue($revision_field->hasAttribute('disabled'));
        $this->assertTrue($revision_field->isChecked());
    }

    /**
     * Tests that revision log messages can be required.
     */
    public function testRevisionLogRequirement()
    {
        // Create a content type and configure it to require revisions and log.
        $this->drupalCreateContentType(
            [
            'type' => 'article',
            'name' => 'Article',
            ]
        );

        $config = $this->config('require_revision.settings');
        $config->set('revision_required_node', ['article' => 'article']);
        $config->set('revision_log_required_node', ['article' => 'article']);
        $config->save();

        // Create a user who can create articles.
        $user = $this->drupalCreateUser(
            [
            'create article content',
            'edit own article content',
            ]
        );
        $this->drupalLogin($user);

        // Create a node.
        $node = $this->drupalCreateNode(
            [
            'type' => 'article',
            'title' => 'Test Article',
            ]
        );

        // Visit the edit form.
        $this->drupalGet('node/' . $node->id() . '/edit');

        // The revision log field should be required.
        $page = $this->getSession()->getPage();
        $revision_log_field = $page->find(
            'css',
            'textarea[name="revision_log[0][value]"]'
        );
        $this->assertNotNull($revision_log_field);
        $this->assertTrue($revision_log_field->hasAttribute('required'));
    }

}