<?php

namespace Drupal\require_revision\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Require Revision settings.
 */
class RequireRevisionSettingsForm extends ConfigFormBase {

  /**
   * Constructs a RequireRevisionSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory service.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config
   *   The typed config manager service.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config,
    protected EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($config_factory, $typed_config);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['require_revision.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'require_revision_settings';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Get all block content types.
    $block_types = $this->entityTypeManager
      ->getStorage('block_content_type')
      ->loadMultiple();
    $block_type_options = [];
    foreach ($block_types as $id => $block_type) {
      $block_type_options[$id] = $block_type->label();
    }

    // Get all content types (nodes).
    $content_types = $this->entityTypeManager
      ->getStorage('node_type')
      ->loadMultiple();
    $content_type_options = [];
    foreach ($content_types as $id => $content_type) {
      $content_type_options[$id] = $content_type->label();
    }

    // Get all taxonomy vocabularies.
    $vocabularies = $this->entityTypeManager
      ->getStorage('taxonomy_vocabulary')
      ->loadMultiple();
    $vocabulary_options = [];
    foreach ($vocabularies as $id => $vocabulary) {
      $vocabulary_options[$id] = $vocabulary->label();
    }

    $form['description'] = [
      '#markup' => '<p>' . $this->t('Select which entity types should require revisions and revision log messages when editing.') . '</p>',
    ];

    // Block types section.
    if (!empty($block_type_options)) {
      $form['block_types_section'] = [
        '#type' => 'details',
        '#title' => $this->t('Block Types'),
        '#open' => FALSE,
      ];
      $this->addRevisionFields(
        $form['block_types_section'],
        'block',
        $block_type_options
      );
    }

    // Content types section.
    if (!empty($content_type_options)) {
      $form['content_types_section'] = [
        '#type' => 'details',
        '#title' => $this->t('Content Types'),
        '#open' => FALSE,
      ];
      $this->addRevisionFields(
        $form['content_types_section'],
        'node',
        $content_type_options
      );
    }

    // Taxonomy vocabularies section.
    if (!empty($vocabulary_options)) {
      $form['taxonomy_section'] = [
        '#type' => 'details',
        '#title' => $this->t('Taxonomy Vocabularies'),
        '#open' => FALSE,
      ];
      $this->addRevisionFields(
        $form['taxonomy_section'],
        'taxonomy',
        $vocabulary_options
      );
    }

    if (empty($block_type_options)
      && empty($content_type_options)
      && empty($vocabulary_options)
    ) {
      $form['message'] = [
        '#markup' => $this->t('No entity types are currently defined.'),
      ];
    }
    else {
      // Attach JavaScript library and pass entity types to JavaScript.
      $all_entity_types = [
        'block' => array_keys($block_type_options),
        'node' => array_keys($content_type_options),
        'taxonomy' => array_keys($vocabulary_options),
      ];

      $form['#attached']['library'][] = 'require_revision/admin';
      $form['#attached']['drupalSettings']['requireRevision'] = [
        'entityTypes' => $all_entity_types,
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Helper function to add revision fields for an entity type.
   *
   * @param array $form
   *   The form element to add fields to.
   * @param string $entity_type
   *   The entity type key (block, node, or taxonomy).
   * @param array $options
   *   The available entity type options.
   */
  protected function addRevisionFields(array &$form, $entity_type, array $options) {
    // Single "Select all" button for the entire section.
    $form['select_all_' . $entity_type] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Select all'),
      '#weight' => -10,
    ];

    $form['revision_required_' . $entity_type] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Requiring revisions'),
      '#description' => $this->t('Select the types that should require a new revision to be created when edited.'),
      '#options' => $options,
      '#config_target' => 'require_revision.settings:revision_required_' . $entity_type,
    ];

    $form['revision_log_required_' . $entity_type] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Requiring revision log messages'),
      '#description' => $this->t('Select the types that should require a revision log message. This only applies when the type is also set to require revisions above.'),
      '#options' => $options,
      '#config_target' => 'require_revision.settings:revision_log_required_' . $entity_type,
    ];

    // Add #states to disable each revision log checkbox.
    foreach ($options as $type_id => $label) {
      $form['revision_log_required_' . $entity_type][$type_id]['#states'] = [
        'enabled' => [
          ':input[name="revision_required_' . $entity_type . '[' . $type_id . ']"]' => ['checked' => TRUE],
        ],
      ];
    }
  }

}
