<?php

namespace Drupal\require_revision\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Hook implementations for the Require Revision module.
 */
class RequireRevisionHooks {

  use StringTranslationTrait;

  /**
   * Constructs a RequireRevisionHooks object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
  ) {
  }

  /**
   * Implements hook_form_alter().
   */
  #[Hook('form_alter')]
  public function formAlter(&$form, FormStateInterface $form_state, $form_id): void {
    // Handle block content forms.
    if (str_starts_with($form_id, 'block_content_') && str_ends_with($form_id, '_form')) {
      $entity = $form_state->getFormObject()->getEntity();
      $this->applySettings($form, $entity, 'block');
    }

    // Handle node (content type) forms.
    if (str_contains($form_id, 'node_') && str_ends_with($form_id, '_form')) {
      $entity = $form_state->getFormObject()->getEntity();
      $this->applySettings($form, $entity, 'node');
    }

    // Handle taxonomy term forms.
    if (str_starts_with($form_id, 'taxonomy_term_') && str_ends_with($form_id, '_form')) {
      $entity = $form_state->getFormObject()->getEntity();
      $this->applySettings($form, $entity, 'taxonomy');
    }
  }

  /**
   * Implements hook_entity_presave().
   */
  #[Hook('entity_presave')]
  public function entityPresave(EntityInterface $entity): void {
    // Only apply to existing entities, not new ones.
    if ($entity->isNew()) {
      return;
    }

    $entity_type_id = $entity->getEntityTypeId();

    // Map entity type IDs to our config keys.
    $entity_type_map = [
      'block_content' => 'block',
      'node' => 'node',
      'taxonomy_term' => 'taxonomy',
    ];

    if (!isset($entity_type_map[$entity_type_id])) {
      return;
    }

    $config = $this->configFactory->get('require_revision.settings');
    $config_key = $entity_type_map[$entity_type_id];
    $revision_required = $config->get('revision_required_' . $config_key) ?: [];
    $revision_log_required = $config->get('revision_log_required_' . $config_key) ?: [];

    $bundle = $entity->bundle();

    // Ensure revision is created if required.
    if (isset($revision_required[$bundle]) && $revision_required[$bundle]) {
      $entity->setNewRevision(TRUE);

      // Check if revision log message is required and empty.
      if (isset($revision_log_required[$bundle]) && $revision_log_required[$bundle]) {
        $revision_log = $entity->getRevisionLogMessage();

        if (empty($revision_log)) {
          // Set a default message if empty (as a fallback).
          $entity->setRevisionLogMessage($this->t('Required revision log message was not provided.'));
        }
      }
    }
  }

  /**
   * Helper function to apply revision settings to entity forms.
   *
   * @param array $form
   *   The form array to modify.
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity being edited.
   * @param string $entity_type
   *   The entity type key (block, node, or taxonomy).
   */
  protected function applySettings(array &$form, EntityInterface $entity, string $entity_type): void {
    // Only apply to existing entities, not new ones.
    if ($entity->isNew()) {
      return;
    }

    $config = $this->configFactory->get('require_revision.settings');
    $revision_required = $config->get('revision_required_' . $entity_type) ?: [];
    $revision_log_required = $config->get('revision_log_required_' . $entity_type) ?: [];

    if ($entity && $entity->bundle()) {
      $bundle = $entity->bundle();

      // Make revision checkbox required and checked if configured.
      if (isset($revision_required[$bundle]) && $revision_required[$bundle]) {
        if (isset($form['revision'])) {
          $form['revision']['#default_value'] = TRUE;
          $form['revision']['#disabled'] = TRUE;
          $form['revision']['#description'] = $this->t('Revisions are required for this type.');
        }

        // For nodes, handle revision_information fieldset.
        if ($entity_type === 'node' && isset($form['revision_information'])) {
          if (isset($form['revision_information']['revision'])) {
            $form['revision_information']['revision']['#default_value'] = TRUE;
            $form['revision_information']['revision']['#disabled'] = TRUE;
            $form['revision_information']['revision']['#description'] = $this->t('Revisions are required for this content type.');
          }
        }

        // Make revision log message required if configured.
        if (isset($revision_log_required[$bundle]) && $revision_log_required[$bundle]) {
          // Block content.
          if (isset($form['revision_log'])) {
            $form['revision_log']['widget'][0]['value']['#required'] = TRUE;
          }

          // Node.
          if (isset($form['revision_information']['revision_log'])) {
            $form['revision_information']['revision_log']['widget'][0]['value']['#required'] = TRUE;
          }

          // Taxonomy term.
          if (isset($form['revision_log_message'])) {
            $form['revision_log_message']['widget'][0]['value']['#required'] = TRUE;
          }
        }
      }
    }
  }

}
