<?php

declare(strict_types=1);

namespace Drupal\Tests\require_revision_log_message\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests module installation and uninstallation.
 *
 * @group require_revision_log_message
 */
class InstallTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'field',
    'text',
  ];

  /**
   * Tests that the module can be installed and uninstalled.
   */
  public function testModuleInstallUninstall(): void {
    $module_installer = \Drupal::service('module_installer');

    // Install the module.
    $module_installer->install(['require_revision_log_message']);
    $this->assertTrue(\Drupal::moduleHandler()->moduleExists('require_revision_log_message'));

    // Verify configuration exists.
    $config = \Drupal::config('require_revision_log_message.adminsettings');
    $this->assertNotNull($config);
    $this->assertIsArray($config->get('content_types'));
    $this->assertFalse($config->get('require_for_new_nodes'));

    // Uninstall the module.
    $module_installer->uninstall(['require_revision_log_message']);
    $this->assertFalse(\Drupal::moduleHandler()->moduleExists('require_revision_log_message'));
  }

  /**
   * Tests the update hook that adds require_for_new_nodes setting.
   */
  public function testUpdate20001(): void {
    $module_installer = \Drupal::service('module_installer');
    $module_installer->install(['require_revision_log_message']);

    // Get the config and remove the require_for_new_nodes setting.
    $config = \Drupal::configFactory()->getEditable('require_revision_log_message.adminsettings');
    $config->set('content_types', ['article' => 'article']);
    $config->clear('require_for_new_nodes');
    $config->save();

    // Verify the setting doesn't exist.
    $config = \Drupal::config('require_revision_log_message.adminsettings');
    $this->assertNull($config->get('require_for_new_nodes'));

    // Run the update hook.
    require_once dirname(__FILE__, 4) . '/require_revision_log_message.install';
    $result = require_revision_log_message_update_20001();

    // Verify the setting was added with correct default value.
    $config = \Drupal::config('require_revision_log_message.adminsettings');
    $this->assertFalse($config->get('require_for_new_nodes'));
    $this->assertStringContainsString('require_for_new_nodes', (string) $result);

    // Verify other config wasn't changed.
    $this->assertEquals(['article' => 'article'], $config->get('content_types'));
  }

}
