<?php

declare(strict_types=1);

namespace Drupal\Tests\require_revision_log_message\Functional;

use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\BrowserTestBase;

/**
 * Functional tests for Require Revision Log Message module.
 *
 * @group require_revision_log_message
 */
class RequireRevisionLogMessageTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'field',
    'text',
    'user',
    'require_revision_log_message',
  ];

  /**
   * A user with administrative permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A user with content editing permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $editorUser;

  /**
   * A user with bypass permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $bypassUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create content type.
    $node_type = NodeType::create([
      'type' => 'article',
      'name' => 'Article',
      'new_revision' => TRUE,
    ]);
    $node_type->save();

    // Create users.
    $this->adminUser = $this->drupalCreateUser([
      'administer require_revision_log_message',
      'administer content types',
      'create article content',
      'edit any article content',
      'access content overview',
    ]);

    $this->editorUser = $this->drupalCreateUser([
      'create article content',
      'edit any article content',
    ]);

    $this->bypassUser = $this->drupalCreateUser([
      'create article content',
      'edit any article content',
      'bypass require_revision_log_message',
    ]);
  }

  /**
   * Tests the settings form.
   */
  public function testSettingsForm(): void {
    $this->drupalLogin($this->adminUser);

    // Visit the settings page.
    $this->drupalGet('admin/config/require-revision-log/adminsettings');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Content Types');
    $this->assertSession()->pageTextContains('Require revision log message for new nodes');

    // Test form submission.
    $edit = [
      'content_types[article]' => 'article',
      'require_for_new_nodes' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify configuration was saved.
    $config = $this->config('require_revision_log_message.adminsettings');
    $this->assertArrayHasKey('article', $config->get('content_types'));
    $this->assertTrue($config->get('require_for_new_nodes'));
  }

  /**
   * Tests that revision log is required on existing nodes.
   */
  public function testRevisionLogRequiredOnExistingNode(): void {
    // Configure module to require revision log for article.
    $config = $this->config('require_revision_log_message.adminsettings');
    $config->set('content_types', ['article' => 'article'])
      ->set('require_for_new_nodes', FALSE)
      ->save();

    // Create a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'uid' => $this->editorUser->id(),
    ]);
    $node->save();

    // Login as editor and try to edit the node.
    $this->drupalLogin($this->editorUser);
    $this->drupalGet('node/' . $node->id() . '/edit');

    // Check that revision log is required.
    $revision_log_field = $this->assertSession()->fieldExists('revision_log[0][value]');
    $this->assertTrue($revision_log_field->hasAttribute('required'));

    // Try to submit without revision log.
    $edit = [
      'title[0][value]' => 'Updated Article',
      'revision_log[0][value]' => '',
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('Revision log message field is required.');
  }

  /**
   * Tests that revision log is not required for new nodes by default.
   */
  public function testRevisionLogNotRequiredOnNewNodeByDefault(): void {
    // Configure module to require revision log for article but not new nodes.
    $config = $this->config('require_revision_log_message.adminsettings');
    $config->set('content_types', ['article' => 'article'])
      ->set('require_for_new_nodes', FALSE)
      ->save();

    $this->drupalLogin($this->editorUser);
    $this->drupalGet('node/add/article');

    // Check that revision log is NOT required for new nodes.
    $revision_log_field = $this->assertSession()->fieldExists('revision_log[0][value]');
    $this->assertFalse($revision_log_field->hasAttribute('required'));

    // Should be able to create without revision log.
    $edit = [
      'title[0][value]' => 'New Article Without Revision Log',
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('Article New Article Without Revision Log has been created.');
  }

  /**
   * Tests that revision log is required for new nodes when configured.
   */
  public function testRevisionLogRequiredOnNewNodeWhenConfigured(): void {
    // Configure module to require revision log for article including new nodes.
    $config = $this->config('require_revision_log_message.adminsettings');
    $config->set('content_types', ['article' => 'article'])
      ->set('require_for_new_nodes', TRUE)
      ->save();

    $this->drupalLogin($this->editorUser);
    $this->drupalGet('node/add/article');

    // Check that revision log IS required for new nodes.
    $revision_log_field = $this->assertSession()->fieldExists('revision_log[0][value]');
    $this->assertTrue($revision_log_field->hasAttribute('required'));

    // Try to submit without revision log.
    $edit = [
      'title[0][value]' => 'New Article',
      'revision_log[0][value]' => '',
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('Revision log message field is required.');

    // Submit with revision log.
    $edit = [
      'title[0][value]' => 'New Article With Revision Log',
      'revision_log[0][value]' => 'Initial creation',
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('Article New Article With Revision Log has been created.');
  }

  /**
   * Tests that users with bypass permission are not affected.
   */
  public function testBypassPermission(): void {
    // Configure module to require revision log for article.
    $config = $this->config('require_revision_log_message.adminsettings');
    $config->set('content_types', ['article' => 'article'])
      ->set('require_for_new_nodes', TRUE)
      ->save();

    // Create a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'uid' => $this->bypassUser->id(),
    ]);
    $node->save();

    // Login as bypass user.
    $this->drupalLogin($this->bypassUser);
    $this->drupalGet('node/' . $node->id() . '/edit');

    // Check that revision log is NOT required.
    $revision_log_field = $this->assertSession()->fieldExists('revision_log[0][value]');
    $this->assertFalse($revision_log_field->hasAttribute('required'));

    // Should be able to submit without revision log.
    $edit = [
      'title[0][value]' => 'Updated Without Revision Log',
      'revision_log[0][value]' => '',
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('Article Updated Without Revision Log has been updated.');
  }

  /**
   * Tests that revision checkbox is disabled when required.
   */
  public function testRevisionCheckboxDisabled(): void {
    // Configure module to require revision log for article.
    $config = $this->config('require_revision_log_message.adminsettings');
    $config->set('content_types', ['article' => 'article'])
      ->set('require_for_new_nodes', FALSE)
      ->save();

    // Create a node.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Test Article',
      'uid' => $this->editorUser->id(),
    ]);
    $node->save();

    // Login as editor and visit the edit form.
    $this->drupalLogin($this->editorUser);
    $this->drupalGet('node/' . $node->id() . '/edit');

    // Check that revision checkbox is disabled.
    $revision_checkbox = $this->assertSession()->fieldExists('revision');
    $this->assertTrue($revision_checkbox->hasAttribute('disabled'));
  }

  /**
   * Tests that not configured content types are not affected.
   */
  public function testNotConfiguredContentType(): void {
    // Create another content type.
    NodeType::create([
      'type' => 'page',
      'name' => 'Page',
      'new_revision' => TRUE,
    ])->save();

    // Give permissions to create pages.
    $this->editorUser->addRole($this->drupalCreateRole([
      'create page content',
      'edit any page content',
    ]));
    $this->editorUser->save();

    // Configure module to require revision log only for article.
    $config = $this->config('require_revision_log_message.adminsettings');
    $config->set('content_types', ['article' => 'article'])
      ->set('require_for_new_nodes', TRUE)
      ->save();

    // Create a page node.
    $node = Node::create([
      'type' => 'page',
      'title' => 'Test Page',
      'uid' => $this->editorUser->id(),
    ]);
    $node->save();

    // Login as editor and visit the edit form.
    $this->drupalLogin($this->editorUser);
    $this->drupalGet('node/' . $node->id() . '/edit');

    // Check that revision log is NOT required for pages.
    $revision_log_field = $this->assertSession()->fieldExists('revision_log[0][value]');
    $this->assertFalse($revision_log_field->hasAttribute('required'));

    // Should be able to submit without revision log.
    $edit = [
      'title[0][value]' => 'Updated Page',
      'revision_log[0][value]' => '',
    ];
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('Page Updated Page has been updated.');
  }

  /**
   * Tests the help page.
   */
  public function testHelpPage(): void {
    // Install help module.
    \Drupal::service('module_installer')->install(['help']);
    $this->rebuildContainer();

    // Create user with help access.
    $user = $this->drupalCreateUser([
      'access administration pages',
      'access help pages',
      'administer require_revision_log_message',
    ]);
    $this->drupalLogin($user);

    $this->drupalGet('admin/help/require_revision_log_message');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('About');
    $this->assertSession()->pageTextContains('requires that the revision log be filled in');
  }

  /**
   * Tests access to the settings form.
   */
  public function testSettingsFormAccess(): void {
    // Try to access without permission.
    $this->drupalLogin($this->editorUser);
    $this->drupalGet('admin/config/require-revision-log/adminsettings');
    $this->assertSession()->statusCodeEquals(403);

    // Access with permission.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/require-revision-log/adminsettings');
    $this->assertSession()->statusCodeEquals(200);
  }

}
