<?php

declare(strict_types=1);

namespace Drupal\Tests\require_revision_log_message\Kernel;

use Drupal\Core\Form\FormState;
use Drupal\KernelTests\KernelTestBase;
use Drupal\node\Entity\NodeType;

/**
 * Kernel tests for the SettingsForm.
 *
 * @group require_revision_log_message
 */
class SettingsFormTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'node',
    'user',
    'field',
    'text',
    'require_revision_log_message',
  ];

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('node');
    $this->installSchema('node', ['node_access']);
    $this->installConfig(['system', 'node', 'field', 'require_revision_log_message']);

    // Create content types.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    NodeType::create([
      'type' => 'page',
      'name' => 'Page',
    ])->save();

    $this->formBuilder = $this->container->get('form_builder');
  }

  /**
   * Tests that the configuration form can be built.
   */
  public function testFormBuilds(): void {
    $form = $this->formBuilder->getForm('\Drupal\require_revision_log_message\Form\SettingsForm');

    $this->assertArrayHasKey('content_types', $form);
    $this->assertEquals('checkboxes', $form['content_types']['#type']);
    $this->assertArrayHasKey('require_for_new_nodes', $form);
    $this->assertEquals('checkbox', $form['require_for_new_nodes']['#type']);
  }

  /**
   * Tests form submission saves configuration correctly.
   */
  public function testFormSubmission(): void {
    // Build the form.
    $form_state = new FormState();

    // Set form values as they would come from a real form submission.
    $form_state->setValues([
      'content_types' => ['article' => 'article', 'page' => 0],
      'require_for_new_nodes' => TRUE,
    ]);

    // Get the form object using the class resolver.
    $form_object = $this->container->get('class_resolver')
      ->getInstanceFromDefinition('\Drupal\require_revision_log_message\Form\SettingsForm');

    // Build and submit the form.
    $form = [];
    $form = $form_object->buildForm($form, $form_state);
    $form_object->submitForm($form, $form_state);

    // Check the saved configuration.
    $config = $this->config('require_revision_log_message.adminsettings');
    $content_types = $config->get('content_types') ?? [];

    $this->assertArrayHasKey('article', $content_types);
    $this->assertEquals('article', $content_types['article']);
    $this->assertArrayNotHasKey('page', $content_types);
    $this->assertTrue($config->get('require_for_new_nodes'));
  }

  /**
   * Tests default configuration values.
   */
  public function testDefaultConfiguration(): void {
    $config = $this->config('require_revision_log_message.adminsettings');

    $this->assertIsArray($config->get('content_types'));
    $this->assertEmpty($config->get('content_types'));
    $this->assertFalse($config->get('require_for_new_nodes'));
  }

  /**
   * Tests configuration schema.
   */
  public function testConfigurationSchema(): void {
    $config = $this->config('require_revision_log_message.adminsettings');

    // Set some test values.
    $config->set('content_types', ['article' => 'article'])
      ->set('require_for_new_nodes', TRUE)
      ->save();

    // Verify the configuration can be saved and loaded.
    $saved_config = $this->config('require_revision_log_message.adminsettings');
    $this->assertEquals(['article' => 'article'], $saved_config->get('content_types'));
    $this->assertTrue($saved_config->get('require_for_new_nodes'));
  }

}
