<?php

declare(strict_types=1);

namespace Drupal\Tests\require_revision_log_message\Unit\Form;

use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\node\NodeTypeInterface;
use Drupal\require_revision_log_message\Form\SettingsForm;
use Drupal\Tests\UnitTestCase;

/**
 * Unit tests for the SettingsForm.
 *
 * @coversDefaultClass \Drupal\require_revision_log_message\Form\SettingsForm
 * @group require_revision_log_message
 */
class SettingsFormTest extends UnitTestCase {

  /**
   * The mocked entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The form instance.
   *
   * @var \Drupal\require_revision_log_message\Form\SettingsForm
   */
  protected $form;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->form = new SettingsForm($this->entityTypeManager);

    // Set up string translation mock.
    $string_translation = $this->createMock(TranslationInterface::class);
    $string_translation->method('translate')
      ->willReturnCallback(function ($string) {
        return $string;
      });
    $this->form->setStringTranslation($string_translation);
  }

  /**
   * Tests the form ID.
   *
   * @covers ::getFormId
   */
  public function testGetFormId(): void {
    $this->assertEquals('require_revision_log_settings_form', $this->form->getFormId());
  }

  /**
   * Tests getting editable config names.
   *
   * @covers ::getEditableConfigNames
   */
  public function testGetEditableConfigNames(): void {
    // Use reflection to access protected method.
    $reflection = new \ReflectionClass($this->form);
    $method = $reflection->getMethod('getEditableConfigNames');
    $method->setAccessible(TRUE);

    $expected = ['require_revision_log_message.adminsettings'];
    $this->assertEquals($expected, $method->invoke($this->form));
  }

  /**
   * Tests that the form constructor accepts entity type manager.
   *
   * @covers ::__construct
   */
  public function testConstructor(): void {
    $form = new SettingsForm($this->entityTypeManager);
    $this->assertInstanceOf(SettingsForm::class, $form);
  }

  /**
   * Tests buildForm builds the expected form structure.
   *
   * @covers ::buildForm
   */
  public function testBuildForm(): void {
    // Mock node types.
    $nodeType1 = $this->createMock(NodeTypeInterface::class);
    $nodeType1->method('id')->willReturn('article');
    $nodeType1->method('label')->willReturn('Article');

    $nodeType2 = $this->createMock(NodeTypeInterface::class);
    $nodeType2->method('id')->willReturn('page');
    $nodeType2->method('label')->willReturn('Page');

    $nodeTypes = [
      'article' => $nodeType1,
      'page' => $nodeType2,
    ];

    // Mock node type storage.
    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('loadMultiple')
      ->willReturn($nodeTypes);

    $this->entityTypeManager->method('getStorage')
      ->with('node_type')
      ->willReturn($storage);

    // Mock config.
    $config = $this->getConfigFactoryStub([
      'require_revision_log_message.adminsettings' => [
        'content_types' => ['article' => 'article'],
        'require_for_new_nodes' => TRUE,
      ],
    ]);

    $this->form->setConfigFactory($config);

    // Build the form.
    $form_state = $this->createMock(FormStateInterface::class);
    $form = $this->form->buildForm([], $form_state);

    // Assert form structure.
    $this->assertArrayHasKey('content_types', $form);
    $this->assertEquals('checkboxes', $form['content_types']['#type']);
    $this->assertArrayHasKey('article', $form['content_types']['#options']);
    $this->assertArrayHasKey('page', $form['content_types']['#options']);
    $this->assertEquals('Article', $form['content_types']['#options']['article']);
    $this->assertEquals('Page', $form['content_types']['#options']['page']);

    $this->assertArrayHasKey('require_for_new_nodes', $form);
    $this->assertEquals('checkbox', $form['require_for_new_nodes']['#type']);
    $this->assertTrue($form['require_for_new_nodes']['#default_value']);
  }

}
