<?php

namespace Drupal\required_api;

use Drupal\Core\Entity\ContentEntityFormInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\EntityReferenceFieldItemListInterface;
use Drupal\Core\Form\FormErrorHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Provides an entity form error handler that clears unnecessary form errors.
 */
class RequiredApiFormErrorHandler implements FormErrorHandlerInterface {

  /**
   * Constructs a RequiredApiFormErrorHandler object.
   *
   * @param \Drupal\Core\Form\FormErrorHandlerInterface $formErrorHandler
   *   The inner form error handler service.
   * @param \Drupal\required_api\RequiredManager $requiredManager
   *   The required manager.
   */
  public function __construct(
    protected FormErrorHandlerInterface $formErrorHandler,
    protected RequiredManager $requiredManager,
  ) {
  }

  /**
   * {@inheritdoc}
   */
  public function handleFormErrors(array &$form, FormStateInterface $form_state) {
    $entityForm = $form_state->getFormObject();
    if (!$entityForm instanceof ContentEntityFormInterface) {
      $this->formErrorHandler->handleFormErrors($form, $form_state);
      return $this;
    }

    // Get an entity with up-to-date values.
    // We clone the form state to make sure any changes are not persisted.
    $temp_form_state = clone $form_state;
    // Mark the entity as not validated to ensure that the entity is
    // revalidated again. This to prevent that entity builders are executed
    // again.
    $temp_form_state->setTemporaryValue('entity_validated', FALSE);
    $entity = $entityForm->buildEntity($form, $temp_form_state);
    assert($entity instanceof ContentEntityInterface);

    // Get the errors and clear them.
    $errors = $form_state->getErrors();
    $form_state->clearErrors();

    foreach ($entity->getFieldDefinitions() as $fieldName => $fieldDefinition) {
      if ($fieldDefinition instanceof BaseFieldDefinition) {
        continue;
      }

      // Determine if the field is required.
      $plugin = $this->requiredManager->getInstance(['field_definition' => $fieldDefinition]);
      $required = $plugin->isRequired($fieldDefinition, $entity);
      if ($required === TRUE) {
        continue;
      }

      // Remove all required error messages for this field.
      $required_error_messages = [
        '@name field is required.',
        'The %field date is required.',
      ];

      foreach ($errors as $name => $message) {
        if ($message instanceof TranslatableMarkup) {
          $rawMessage = $message->getUntranslatedString();
        }
        else {
          $rawMessage = $message;
        }

        if (!in_array($rawMessage, $required_error_messages, TRUE)) {
          continue;
        }

        if ($name !== $fieldName && substr($name, 0, strpos($name, ']')) !== $fieldName) {
          continue;
        }

        // If the error name starts with the field name, we assume it is related
        // to this field.
        $fieldPath = $this->parseFieldPath($name);
        if ($fieldPath && $this->isFieldActuallyRequired($fieldPath, $entity)) {
          // If the field is actually required, we skip it.
          continue;
        }

        unset($errors[$name]);
      }
    }

    // Re-add the remaining errors.
    foreach ($errors as $name => $message) {
      $form_state->setErrorByName($name, $message);
    }

    $this->formErrorHandler->handleFormErrors($form, $form_state);
    return $this;
  }

  /**
   * Parses the field path from an error name.
   */
  private function parseFieldPath(string $error_name): array {
    $parts = explode('][', $error_name);
    $path = [];
    foreach ($parts as $part) {
      $part = trim($part, '[]');
      if (empty($part) || is_numeric($part) || $part === 'subform') {
        continue;
      }
      // Skip parts that are not relevant for field paths.
      // We assume that 'value', 'target_id', 'alt', and 'title' are the only
      // parts that are not part of the field path.
      if (in_array($part, ['value', 'target_id', 'alt', 'title'])) {
        break;
      }
      $path[] = $part;
    }
    return $path;
  }

  /**
   * Checks if a field is actually required.
   */
  private function isFieldActuallyRequired(array $fieldPath, ContentEntityInterface $entity): bool {
    $currentEntity = $entity;

    foreach ($fieldPath as $fieldName) {
      if (!$currentEntity->hasField($fieldName)) {
        return FALSE;
      }

      $field = $currentEntity->get($fieldName);
      if ($field instanceof EntityReferenceFieldItemListInterface) {
        $currentEntity = $field->entity;
        if (!$currentEntity instanceof ContentEntityInterface) {
          return FALSE;
        }
      }
      else {
        // If the field is not an entity reference, we assume it is a field on
        // the current entity.
        $fieldDefinition = $currentEntity->getFieldDefinition($fieldName);
        $plugin = $this->requiredManager->getInstance(['field_definition' => $fieldDefinition]);
        return $plugin->isRequired($fieldDefinition, $currentEntity);
      }
    }

    return FALSE;
  }

}
