<?php

namespace Drupal\required_api;

use Drupal\Component\Plugin\FallbackPluginManagerInterface;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Plugin\DefaultPluginManager;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\required_api\Annotation\Required as RequiredAnnotation;
use Drupal\required_api\Attribute\Required;
use Drupal\required_api\Plugin\Required\Broken;
use Drupal\required_api\Plugin\RequiredPluginInterface;
use Psr\Log\LoggerInterface;

/**
 * Manages required by role plugins.
 */
class RequiredManager extends DefaultPluginManager implements FallbackPluginManagerInterface {

  use StringTranslationTrait;

  /**
   * Constructs a new RequiredManager object.
   *
   * @param \Traversable $namespaces
   *   An object that implements \Traversable which contains the root paths
   *   keyed by the corresponding namespace to look for plugin implementations.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cacheBackend
   *   Cache backend instance to use.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   */
  public function __construct(
    \Traversable $namespaces,
    CacheBackendInterface $cacheBackend,
    ModuleHandlerInterface $moduleHandler,
    protected ConfigFactoryInterface $configFactory,
    protected LoggerInterface $logger,
  ) {
    parent::__construct(
      'Plugin/Required',
      $namespaces,
      $moduleHandler,
      RequiredPluginInterface::class,
      Required::class,
      RequiredAnnotation::class,
    );
    $this->setCacheBackend($cacheBackend, 'required_api_required_plugins');
  }

  /**
   * {@inheritdoc}
   */
  public function getInstance(array $options) {
    assert($options['field_definition'] instanceof FieldDefinitionInterface);
    $options['plugin_id'] ??= $this->getPluginId($options['field_definition']);

    return $this->createInstance($options['plugin_id'], $options);
  }

  /**
   * {@inheritdoc}
   */
  public function getFallbackPluginId($plugin_id, array $configuration = []) {
    return Broken::ID;
  }

  /**
   * Gets the plugin_id for this field definition, fallback to system default.
   *
   * @param \Drupal\Core\Field\FieldDefinitionInterface $field
   *   A field instance.
   *
   * @return string|null
   *   The plugin id.
   */
  public function getPluginId(FieldDefinitionInterface $field): ?string {
    return $field->getThirdPartySetting('required_api', 'required_plugin', $this->getDefaultPluginId());
  }

    /**
   * Gets the default plugin_id for the system.
   *
   * @return string|null
   *   The plugin id.
   */
  public function getDefaultPluginId(): ?string {
    return $this->configFactory
      ->get('required_api.plugins')
      ->get('default_plugin');
  }

  /**
   * Provides the definition ids.
   */
  public function getDefinitionsIds(): array {
    return array_keys($this->getDefinitions());
  }

  /**
   * Provides the definitions as options just to inject to a select element.
   */
  public function getDefinitionsAsOptions(): array {
    $definitions = $this->getDefinitions();
    unset($definitions[Broken::ID]);

    return array_map(function ($definition) {
      return $definition['label'];
    }, $definitions);
  }

  /**
   * {@inheritdoc}
   */
  protected function handlePluginNotFound($plugin_id, array $configuration) {
    $params = [
      '%plugin_id' => $plugin_id,
      '%field_name' => $configuration['field_definition']->id(),
    ];

    $editLinkTemplate = "{$configuration['field_definition']->getTargetEntityTypeId()}-field-edit-form";
    if ($configuration['field_definition']->hasLinkTemplate($editLinkTemplate)) {
      $params['link'] = $configuration['field_definition']->toLink('Edit field', $editLinkTemplate)->toString();
    }

    $this->logger->error('The "%plugin_id" required handler was not found, so the field will be considered as always required. Please reconfigure the "%field_name" field.', $params);

    return parent::handlePluginNotFound($plugin_id, $configuration);
  }

}
