<?php

namespace Drupal\required_api\TypedData;

use Drupal\Core\Entity\Plugin\DataType\EntityAdapter;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\TypedData\Validation\RecursiveValidator;
use Drupal\required_api\RequiredManager;
use Symfony\Component\Validator\ConstraintViolationListInterface;

/**
 * Provides a validator that updates the required state of fields.
 *
 * This class updates the required status in context of programmatic validation.
 */
class RequiredApiRecursiveValidator extends RecursiveValidator {

  /**
   * {@inheritdoc}
   */
  public function validate($value, $constraints = NULL, $groups = NULL): ConstraintViolationListInterface {
    $violations = parent::validate($value, $constraints, $groups);

    $requiredManager = \Drupal::service('plugin.manager.required_api.required');
    assert($requiredManager instanceof RequiredManager);

    foreach ($violations as $i => $violation) {
      $root = $violation->getRoot();
      if (!$root instanceof EntityAdapter) {
        continue;
      }

      $value = $violation->getInvalidValue();
      if (!$value instanceof FieldItemListInterface) {
        continue;
      }

      $fieldDefinition = $value->getFieldDefinition();
      if ($fieldDefinition instanceof BaseFieldDefinition) {
        continue;
      }

      $plugin = $requiredManager->getInstance(['field_definition' => $fieldDefinition]);
      $entity = $root->getValue();
      $required = $plugin->isRequired($fieldDefinition, $entity);

      if (!$required) {
        $violations->remove($i);
      }
    }

    return $violations;
  }

}
