<?php

namespace Drupal\required_if_filled\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Configuration form for the "Required If Field Has Value" module.
 *
 * Allows site administrators to define rules that make a field required
 * only if another field has a value. Rules are defined per bundle in the
 * format: bundle|source_field|required_field.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   *
   * Returns the configuration objects that will be editable.
   *
   * @return array
   *   An array of configuration object names.
   */
  protected function getEditableConfigNames() {
    return ['required_if_filled.settings'];
  }

  /**
   * {@inheritdoc}
   *
   * Returns the unique ID of this form.
   *
   * @return string
   *   The form ID.
   */
  public function getFormId() {
    return 'required_if_filled_settings_form';
  }

  /**
   * {@inheritdoc}
   *
   * Builds the configuration form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   *
   * @return array
   *   The form array with added configuration fields.
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('required_if_filled.settings');

    $form['rules'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Rules'),
      '#description' => $this->t('One rule per line. Format: bundle|source_field|required_field'),
      '#default_value' => implode("\n", $config->get('rules') ?? []),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   *
   * Handles form submission for the configuration form.
   * Saves the rules entered by the administrator to configuration.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current state of the form.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $rules = array_filter(array_map('trim', explode("\n", $form_state->getValue('rules'))));

    $this->config('required_if_filled.settings')
      ->set('rules', $rules)
      ->save();

    parent::submitForm($form, $form_state);
  }

}
