<?php

namespace Drupal\resend_register_mail\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Action\ActionBase;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Defines an action to resend the registration or welcome email to users.
 */
#[Action(
  id: "resend_register_mail_action",
  label: new TranslatableMarkup("Resend registration / welcome email to selected users"),
  type: "user"
)]
class ResendRegisterMail extends ActionBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new ResendRegisterMail object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The configuration factory.
   * @param \Drupal\Core\Session\AccountProxyInterface $account
   *   The current user.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    protected ConfigFactoryInterface $configFactory,
    protected AccountInterface $account,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function execute($account = NULL): void {
    if (empty($account) || empty($account->getEmail())) {
      return;
    }

    $op = 'register_pending_approval';
    if ($account->isActive()) {
      // Determine the user approval method.
      $op = match ($this->configFactory->get('user.settings')
        ->get('register')) {
        UserInterface::REGISTER_ADMINISTRATORS_ONLY => 'register_admin_created',
        default => 'register_no_approval_required',
      };
    }

    _user_mail_notify($op, $account);
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    if ($account === NULL) {
      $account = $this->account;
    }

    /** @var \Drupal\user\UserInterface $object */
    $access = AccessResult::allowedIf($account->hasPermission('administer users') || $account->hasPermission('resend account emails'));
    return $return_as_object ? $access : $access->isAllowed();
  }

}
