<?php

namespace Drupal\resend_register_mail\Hook;

use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityFormInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\user\UserInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Class for the resend_register_mail form hooks.
 */
class FormHooks {

  use DependencySerializationTrait;

  /**
   * Constructor for FormHooks class.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory service for accessing configurations.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   The current user session object.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger service for displaying messages to the user.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerChannelFactory
   *   The logger service for logging messages.
   */
  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected AccountInterface $currentUser,
    protected MessengerInterface $messenger,
    protected LoggerChannelFactoryInterface $loggerChannelFactory,
  ) {}

  /**
   * Function that implements hook_form_BASE_FORM_ID_alter().
   */
  #[Hook('form_user_form_alter')]
  public function resendRegisterMailFormAlter(&$form, FormStateInterface $form_state, $form_id): void {
    if ($form_state->getFormObject() instanceof EntityFormInterface) {
      $entity = $form_state->getFormObject()->getEntity();
      if ($entity instanceof UserInterface) {
        // Determine the user approval method.
        $user_register = $this->configFactory->get('user.settings')->get('register');
        $send_awaiting_approval = match ($user_register) {
          UserInterface::REGISTER_VISITORS_ADMINISTRATIVE_APPROVAL => TRUE,
          default => FALSE,
        };

        if (!$send_awaiting_approval && !$entity->isActive()) {
          return;
        }

        $form['actions']['resend_register_mail']['#type'] = 'submit';
        $form['actions']['resend_register_mail']['#value'] = $entity->isActive() ? t('Resend welcome message') : t('Resend awaiting approval message');
        $form['actions']['resend_register_mail']['#submit'][] = [$this, 'submitForm'];
        $form['actions']['resend_register_mail']['#access'] = $entity->getEmail() && ($this->currentUser->hasPermission('administer users') || $this->currentUser->hasPermission('resend account emails'));
      }
    }
  }

  /**
   * Form submission handler for user form.
   *
   * @see static::resendRegisterMailFormAlter()
   */
  public function submitForm($form, FormStateInterface $form_state): void {
    $account = NULL;
    if ($form_state->getFormObject() instanceof EntityFormInterface) {
      $entity = $form_state->getFormObject()->getEntity();
      if ($entity instanceof UserInterface) {
        $account = $entity;
      }
    }

    $logger = $this->loggerChannelFactory->get('resend_register_mail');

    if ($account instanceof UserInterface) {
      if (!$account->isActive()) {
        $op = 'register_pending_approval';
      }
      else {
        // Determine the user approval method.
        $op = match ($this->configFactory->get('user.settings')->get('register')) {
          UserInterface::REGISTER_ADMINISTRATORS_ONLY => 'register_admin_created',
          default => 'register_no_approval_required',
        };
      }

      // Notify the user via email.
      $mail = _user_mail_notify($op, $account);

      // Log the mail.
      if (!empty($mail)) {
        $logger->notice('Welcome message has been sent to %name at %email.', [
          '%name' => $account->getAccountName(),
          '%email' => $account->getEmail(),
        ]);
        $this->messenger->addMessage(t('Welcome message has been sent to %name at %email', [
          '%name' => $account->getAccountName(),
          '%email' => $account->getEmail(),
        ]));
      }
      else {
        $logger->notice('There was an error sending the welcome message to %name at %email', [
          '%name' => $account->getAccountName(),
          '%email' => $account->getEmail(),
        ]);
        $this->messenger->addMessage(t('There was an error sending the welcome message to %name at %email', [
          '%name' => $account->getAccountName(),
          '%email' => $account->getEmail(),
        ]), 'error');
      }
    }
  }

}
