<?php

namespace Drupal\resend_register_mail\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfirmFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\Core\Url;
use Drupal\user\UserStorageInterface;
use Drupal\user\UserInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides options of emails to send.
 *
 * @internal
 */
class UserMultipleResendEmail extends ConfirmFormBase {

  /**
   * Constructs a new UserMultipleResendEmail object.
   *
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $tempStoreFactory
   *   The temp store factory.
   * @param \Drupal\user\UserStorageInterface $userStorage
   *   The user storage.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    protected readonly PrivateTempStoreFactory $tempStoreFactory,
    protected readonly UserStorageInterface $userStorage,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('tempstore.private'),
      $container->get('entity_type.manager')?->getStorage('user'),
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'user_multiple_resend_email';
  }

  /**
   * {@inheritdoc}
   */
  public function getQuestion(): TranslatableMarkup {
    return $this->t('Select the email type you want to send');
  }

  /**
   * {@inheritdoc}
   */
  public function getCancelUrl(): Url {
    return new Url('entity.user.collection');
  }

  /**
   * {@inheritdoc}
   */
  public function getConfirmText(): TranslatableMarkup {
    return $this->t('Submit');
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription(): string|TranslatableMarkup {
    return '';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    // Retrieve the accounts from temp store to send the emails.
    /** @var \Drupal\user\Entity\User[] $accounts */
    $accounts = $this->tempStoreFactory
      ->get('user_user_operations_resend_email')
      ->get($this->currentUser()->id());
    if (!$accounts) {
      return $this->redirect('entity.user.collection');
    }

    $names = [];
    $user_with_no_mail_id = [];
    $form['accounts'] = ['#tree' => TRUE];
    foreach ($accounts as $account) {
      $uid = $account->id();

      // Check for the mail id.
      if (empty($account->getEmail())) {
        $user_with_no_mail_id[$uid] = $account->label();
      }
      $names[$uid] = $account->label();
      $form['accounts'][$uid] = [
        '#type' => 'hidden',
        '#value' => $uid,
      ];
    }

    $form['account']['names'] = [
      '#theme' => 'item_list',
      '#items' => $names,
    ];

    // Output a notice regarding empty mail id.
    if (!empty($user_with_no_mail_id)) {
      $redirect = (count($accounts) === 1);
      $message = $this->t('The user account %name does not have email id.', ['%name' => $user_with_no_mail_id]);
      $this->messenger()->addMessage($message, $redirect ? MessengerInterface::TYPE_ERROR : MessengerInterface::TYPE_WARNING);
      if ($redirect) {
        return $this->redirect('entity.user.collection');
      }
    }

    $options = [
      'register_admin_created' => $this->t('Welcome message for user created by the admin'),
      'register_no_approval_required' => $this->t('Welcome message when user self-registers.'),
      'register_pending_approval' => $this->t('Welcome message, user pending admin approval.'),
      'password_reset' => $this->t('Password recovery request.'),
    ];

    // Determine default email type based on user configuration.
    $register_setting = $this->configFactory()->get('user.settings')->get('register');
    $default_email_type = match ($register_setting) {
      UserInterface::REGISTER_ADMINISTRATORS_ONLY => 'register_admin_created',
      UserInterface::REGISTER_VISITORS_ADMINISTRATIVE_APPROVAL => 'register_pending_approval',
      default => 'register_no_approval_required',
    };

    $form['message_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Email type'),
      '#description' => $this->t('Choose which type of email you want to send.'),
      '#options' => $options,
      '#required' => TRUE,
      '#default_value' => $default_email_type,
    ];
    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $current_user_id = $this->currentUser()->id();

    // Clear out the accounts from the temp store.
    $this->tempStoreFactory->get('user_user_operations_resend_email')->delete($current_user_id);
    if ($form_state->getValue('confirm')) {
      $email_type = $form_state->getValue('message_type');
      foreach ($form_state->getValue('accounts') as $uid => $value) {
        $account = $this->userStorage->load($uid);
        if ($account && !empty($account->getEmail())) {
          _user_mail_notify($email_type, $account);
        }
      }
      $this->messenger->addMessage(t('Welcome message has been sent to selected users.'));
    }
    $form_state->setRedirect('entity.user.collection');
  }

}
