<?php

namespace Drupal\resend_register_mail\Plugin\Action;

use Drupal\Core\Access\AccessResult;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Action\ActionBase;
use Drupal\Core\Action\Attribute\Action;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides an action to resend registration or welcome emails to users.
 */
#[Action(
  id: "resend_register_mail_action",
  label: new TranslatableMarkup("Resend registration / welcome email to selected users"),
  confirm_form_route_name: "resend_register_mail_action.resend_email",
  type: "user"
)]
class ResendRegisterMail extends ActionBase implements ContainerFactoryPluginInterface {

  /**
   * Constructs a new ResendRegisterMail object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $tempStoreFactory
   *   The tempstore factory.
   * @param \Drupal\Core\Session\AccountInterface $currentUser
   *   Current user.
   */
  public function __construct(
    array $configuration,
    string $plugin_id,
    array $plugin_definition,
    protected readonly PrivateTempStoreFactory $tempStoreFactory,
    protected readonly AccountInterface $currentUser,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('tempstore.private'),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function executeMultiple(array $accounts): void {
    $this->tempStoreFactory->get('user_user_operations_resend_email')->set($this->currentUser->id(), $accounts);
  }

  /**
   * {@inheritdoc}
   */
  public function execute($account = NULL): void {
    $this->executeMultiple([$account]);
  }

  /**
   * {@inheritdoc}
   */
  public function access($object, ?AccountInterface $account = NULL, $return_as_object = FALSE): bool|AccessResultInterface {
    if ($account === NULL) {
      $account = $this->currentUser;
    }

    $access = AccessResult::allowedIf($account->hasPermission('administer users') || $account->hasPermission('resend account emails'));
    return $return_as_object ? $access : $access->isAllowed();
  }

}
