<?php

namespace Drupal\reserved_paths\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;

/**
 * Class AdminForm To administer reserved paths module.
 */
class AdminForm extends ConfigFormBase {

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'reserved_paths_admin_form';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return [
      'reserved_paths.config',
    ];
  }

  /**
   * Get the config name for this entity & bundle.
   *
   * @return string
   *   The compiled config name.
   */
  protected function getConfigName() {
    return 'reserved_paths.config';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $this->helper($form, 'reservedpaths');

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Save'),
      '#attributes' => ['class' => ['btn-primary']],
    ];
    return $form;
  }

  /**
   * Helper function.
   */
  public function helper(&$form, $block_id) {
    $config = $this->config($this->getConfigName());

    $field_name = 'link_paths';
    $form[$field_name] = [
      'link_paths' => [
        '#type' => 'textarea',
        '#title' => $this->t('Link paths'),
        '#description' => $this->t('The paths should be separated by new line. You can use wildcards (*) for pattern matching. Examples:<br/>
          - /admin<br/>
          - /user/*<br/>
          - /api/*/v1<br/>
          - /temp-*'),
        '#default_value' => $config->get($field_name),
        '#rows' => 10,
      ],
    ];

    $error_message_field = 'error_message';
    $form[$error_message_field] = [
      'error_message' => [
        '#type' => 'textfield',
        '#title' => $this->t('Error message'),
        '#description' => $this->t('Custom error message to display when a reserved path is used. Leave empty to use the default message.'),
        '#default_value' => $config->get($error_message_field),
        '#maxlength' => 255,
      ],
    ];

    // Add fieldset for better organization
    $form['settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Wildcard Pattern Examples'),
      '#collapsible' => TRUE,
      '#collapsed' => TRUE,
    ];

    $form['settings']['examples'] = [
      '#type' => 'item',
      '#markup' => $this->t('
        <strong>Wildcard Pattern Examples:</strong><br/>
        <ul>
          <li><code>/admin</code> - Exact match for /admin</li>
          <li><code>/admin/*</code> - Matches /admin/anything</li>
          <li><code>/user/*/edit</code> - Matches /user/123/edit, /user/john/edit, etc.</li>
          <li><code>*/temp</code> - Matches /anything/temp</li>
          <li><code>/api/v*/docs</code> - Matches /api/v1/docs, /api/v2/docs, etc.</li>
        </ul>
        <strong>Note:</strong> Patterns are case-insensitive.
      '),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $field_name = 'link_paths';
    $error_message_field = 'error_message';

    $this->config($this->getConfigName())
      ->set($field_name, $form_state->getValue($field_name))
      ->set($error_message_field, $form_state->getValue($error_message_field))
      ->save();

    $this->messenger()->addMessage($this->t('Reserved paths configuration has been saved.'));
  }

}