<?php

namespace Drupal\resource_conflict\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Datetime\DrupalDateTime;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItemInterface;
use Drupal\field\FieldConfigInterface;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\node\NodeInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\resource_conflict\Event\ConflictDetectedEvent;
use Drupal\resource_conflict\Event\ResourceConflictEvents;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

/**
 * Provides utilities for the Resource Conflict module.
 */
class ResourceConflictManager {

  use StringTranslationTrait;

  /**
   * The name of the managed configuration object.
   */
  private const SETTINGS_NAME = 'resource_conflict.settings';

  /**
   * Constructs the manager.
   */
  public function __construct(
    private readonly ConfigFactoryInterface $configFactory,
    private readonly EntityTypeManagerInterface $entityTypeManager,
    private readonly EntityFieldManagerInterface $entityFieldManager,
    private readonly ModuleHandlerInterface $moduleHandler,
    private readonly MessengerInterface $messenger,
    private readonly EventDispatcherInterface $eventDispatcher,
  ) {}

  /**
   * Returns saved settings for all bundles.
   */
  public function getAllSettings(): array {
    $raw = $this->configFactory->get(self::SETTINGS_NAME)->get('bundles') ?? [];
    return $this->normalizeSettings(is_array($raw) ? $raw : []);
  }

  /**
   * Returns settings for a specific bundle.
   */
  public function getBundleSettings(string $bundle): array {
    $all = $this->getAllSettings();
    if (!isset($all[$bundle])) {
      return [];
    }
    return $all[$bundle] + [
      'default_form_error' => TRUE,
    ];
  }

  /**
   * Determines whether the provided bundle is configured.
   */
  public function isBundleEnabled(string $bundle): bool {
    $settings = $this->getBundleSettings($bundle);
    return !empty($settings['enabled']) && !empty($settings['field_name']);
  }

  /**
   * Saves settings for a bundle.
   */
  public function saveBundleSettings(string $bundle, array $values): void {
    $all = $this->getAllSettings();

    if (empty($values['enabled']) || empty($values['field_name'])) {
      unset($all[$bundle]);
      $this->persistSettings($all);
      return;
    }

    $all[$bundle] = [
      'enabled' => TRUE,
      'field_name' => $values['field_name'],
      'restrict_to_bundle' => !empty($values['restrict_to_bundle']),
      'start_buffer' => trim($values['start_buffer'] ?? ''),
      'end_buffer' => trim($values['end_buffer'] ?? ''),
      'default_form_error' => !empty($values['default_form_error']),
    ];

    $this->persistSettings($all);
  }

  /**
   * Removes any settings for the provided bundle.
   */
  public function deleteBundleSettings(string $bundle): void {
    $all = $this->getAllSettings();
    if (isset($all[$bundle])) {
      unset($all[$bundle]);
      $this->persistSettings($all);
    }
  }

  /**
   * Builds options for compatible fields on the bundle.
   */
  public function getEligibleFieldOptions(string $bundle): array {
    $definitions = $this->entityFieldManager->getFieldDefinitions('node', $bundle);
    $options = [];
    foreach ($definitions as $definition) {
      if ($definition->getFieldStorageDefinition()->isBaseField()) {
        continue;
      }
      if ($this->fieldIsCompatible($definition->getFieldStorageDefinition())) {
        $options[$definition->getName()] = $definition->getLabel();
      }
    }

    natcasesort($options);
    return $options;
  }

  /**
   * Determines if the provided field configuration is compatible.
   */
  public function fieldIsCompatible(FieldStorageDefinitionInterface $definition): bool {
    return $definition->getType() === 'daterange';
  }

  /**
   * Handles a field deletion event.
   */
  public function handleFieldConfigDeletion(FieldConfigInterface $field): void {
    $bundle = $field->getTargetBundle();
    $settings = $this->getBundleSettings($bundle);
    if (!empty($settings) && $settings['field_name'] === $field->getName()) {
      $this->deleteBundleSettings($bundle);
      $this->messenger->addWarning($this->t('Resource conflict detection was disabled for %type because the configured field was removed.', ['%type' => $bundle]));
    }
  }

  /**
   * Handles a field update.
   */
  public function handleFieldConfigUpdate(FieldConfigInterface $field): void {
    $bundle = $field->getTargetBundle();
    $settings = $this->getBundleSettings($bundle);
    if (empty($settings) || $settings['field_name'] !== $field->getName()) {
      return;
    }

    if (!$this->fieldIsCompatible($field->getFieldStorageDefinition())) {
      $this->deleteBundleSettings($bundle);
      $this->messenger->addWarning($this->t('Resource conflict detection was disabled for %type because the configured field is no longer a Date range.', ['%type' => $bundle]));
    }
  }

  /**
   * Loads conflicts for the provided node.
   *
   * @return \Drupal\node\NodeInterface[]
   *   The conflicting nodes keyed by nid.
   */
  public function getConflicts(NodeInterface $node): array {
    $nids = $this->getConflictingNodeIds($node);
    if (empty($nids)) {
      return [];
    }

    /** @var \Drupal\node\NodeStorageInterface $storage */
    $storage = $this->entityTypeManager->getStorage('node');
    $conflicts = $storage->loadMultiple($nids);
    foreach ($conflicts as $conflict) {
      $this->eventDispatcher->dispatch(new ConflictDetectedEvent($node, $conflict), ResourceConflictEvents::CONFLICT_DETECTED);
    }
    $this->moduleHandler->invokeAll('resource_conflict_conflicts_detected', [$node, $conflicts]);

    return $conflicts;
  }

  /**
   * Gathers time spans from the configured date range field.
   */
  protected function getTimeSpans(NodeInterface $node): array {
    $settings = $this->getBundleSettings($node->bundle());
    $field_name = $settings['field_name'] ?? NULL;
    if (!$field_name || !$node->hasField($field_name)) {
      return [];
    }

    $items = $node->get($field_name);
    if ($items->isEmpty()) {
      return [];
    }

    $time_spans = [];

    foreach ($items as $item) {
      /** @var \Drupal\datetime_range\Plugin\Field\FieldType\DateRangeItem $item */
      if ($item->isEmpty()) {
        continue;
      }
      $start_source = $item->get('start_date')->getValue();
      $end_source = $item->get('end_date')->getValue();
      if (!$start_source instanceof DrupalDateTime || !$end_source instanceof DrupalDateTime) {
        continue;
      }

      $start = clone $start_source;
      $end = clone $end_source;
      $start->setTimezone(new \DateTimeZone('UTC'));
      $end->setTimezone(new \DateTimeZone('UTC'));

      if (!empty($settings['start_buffer'])) {
        $start = $this->applyBuffer($start, $settings['start_buffer']);
        if (!$start) {
          continue;
        }
      }

      if (!empty($settings['end_buffer'])) {
        $end = $this->applyBuffer($end, $settings['end_buffer']);
        if (!$end) {
          continue;
        }
      }

      $time_spans[] = [
        'start' => $start->format(DateTimeItemInterface::DATETIME_STORAGE_FORMAT),
        'end' => $end->format(DateTimeItemInterface::DATETIME_STORAGE_FORMAT),
      ];
    }

    return $time_spans;
  }

  /**
   * Applies a buffer string to the provided date.
   */
  protected function applyBuffer(DrupalDateTime $date, string $modifier): ?DrupalDateTime {
    $clone = clone $date;
    try {
      $clone->modify($modifier);
    }
    catch (\Exception) {
      return NULL;
    }

    return $clone;
  }

  /**
   * Finds node IDs that overlap with the provided node.
   *
   * @return int[]
   *   A list of node IDs.
   */
  protected function getConflictingNodeIds(NodeInterface $node): array {
    if (!$this->isBundleEnabled($node->bundle())) {
      return [];
    }

    $settings = $this->getBundleSettings($node->bundle());
    $field_name = $settings['field_name'] ?? NULL;
    if (!$field_name || !$node->hasField($field_name)) {
      return [];
    }

    $time_spans = $this->getTimeSpans($node);
    if (!$time_spans) {
      return [];
    }

    $bundles = $settings['restrict_to_bundle'] ? [$node->bundle()] : $this->getBundlesUsingField($field_name);
    if (empty($bundles)) {
      $bundles = [$node->bundle()];
    }

    /** @var \Drupal\node\NodeStorageInterface $storage */
    $storage = $this->entityTypeManager->getStorage('node');
    $nids = [];

    foreach ($time_spans as $span) {
      $query = $storage->getQuery();
      // Access validation happens higher up in the form/presave hooks.
      // @phpstan-ignore-next-line Drupal's PHPStan rules flag literal FALSE unnecessarily here.
      $query->accessCheck(FALSE);
      $query->condition('type', $bundles, 'IN');
      if ($node->id()) {
        $query->condition('nid', $node->id(), '<>');
      }

      $conditions = $query->orConditionGroup()
        ->condition(
          $query->andConditionGroup()
            ->condition("$field_name.value", $span['start'], '<=')
            ->condition("$field_name.end_value", $span['start'], '>')
        )
        ->condition(
          $query->andConditionGroup()
            ->condition("$field_name.value", $span['end'], '<')
            ->condition("$field_name.end_value", $span['end'], '>=')
        )
        ->condition(
          $query->andConditionGroup()
            ->condition("$field_name.value", $span['start'], '>=')
            ->condition("$field_name.end_value", $span['end'], '<=')
        );

      $query->condition($conditions);

      $result = $query->execute();
      if (!empty($result)) {
        $nids = array_merge($nids, $result);
      }
    }

    $unique = array_unique($nids);
    return array_values(array_map(static fn ($value) => (int) $value, $unique));
  }

  /**
   * Returns bundles that share a field configuration.
   */
  protected function getBundlesUsingField(string $field_name): array {
    $bundles = [];
    foreach ($this->getAllSettings() as $bundle => $settings) {
      if (!empty($settings['enabled']) && ($settings['field_name'] ?? NULL) === $field_name) {
        $bundles[] = $bundle;
      }
    }
    return $bundles;
  }

  /**
   * Normalizes stored settings to an associative array keyed by bundle.
   */
  protected function normalizeSettings(array $raw): array {
    $normalized = [];
    foreach ($raw as $entry) {
      if (!is_array($entry) || empty($entry['bundle'])) {
        continue;
      }
      $bundle = $entry['bundle'];
      unset($entry['bundle']);
      $normalized[$bundle] = $entry;
    }

    return $normalized;
  }

  /**
   * Persists bundle settings to configuration.
   */
  protected function persistSettings(array $settings): void {
    $list = [];
    foreach ($settings as $bundle => $values) {
      $list[] = ['bundle' => $bundle] + $values;
    }

    $this->configFactory->getEditable(self::SETTINGS_NAME)
      ->set('bundles', $list)
      ->save();
  }

}
